<?php
session_start();
require_once '../config/database.php';

// Admin kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$success_message = '';
$error_message = '';

// Banka hesabı ekleme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_bank') {
    $bank_name = trim($_POST['bank_name'] ?? '');
    $account_name = trim($_POST['account_name'] ?? '');
    $iban = trim($_POST['iban'] ?? '');
    $account_number = trim($_POST['account_number'] ?? '');
    $branch_code = trim($_POST['branch_code'] ?? '');
    $currency = trim($_POST['currency'] ?? 'TRY');
    
    if ($bank_name && $account_name && $iban) {
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO bank_accounts (bank_name, account_name, iban, account_number, branch_code, currency) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$bank_name, $account_name, $iban, $account_number, $branch_code, $currency]);
            $success_message = 'Banka hesabı başarıyla eklendi.';
        } catch (Exception $e) {
            $error_message = 'Hata: ' . $e->getMessage();
        }
    } else {
        $error_message = 'Lütfen gerekli alanları doldurun.';
    }
}

// Banka hesabı güncelleme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_bank') {
    $id = (int)($_POST['id'] ?? 0);
    $bank_name = trim($_POST['bank_name'] ?? '');
    $account_name = trim($_POST['account_name'] ?? '');
    $iban = trim($_POST['iban'] ?? '');
    $account_number = trim($_POST['account_number'] ?? '');
    $branch_code = trim($_POST['branch_code'] ?? '');
    $currency = trim($_POST['currency'] ?? 'TRY');
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    if ($id && $bank_name && $account_name && $iban) {
        try {
            $db = getDB();
            $stmt = $db->prepare("UPDATE bank_accounts SET bank_name = ?, account_name = ?, iban = ?, account_number = ?, branch_code = ?, currency = ?, is_active = ? WHERE id = ?");
            $stmt->execute([$bank_name, $account_name, $iban, $account_number, $branch_code, $currency, $is_active, $id]);
            $success_message = 'Banka hesabı başarıyla güncellendi.';
        } catch (Exception $e) {
            $error_message = 'Hata: ' . $e->getMessage();
        }
    } else {
        $error_message = 'Lütfen gerekli alanları doldurun.';
    }
}

// Banka hesabı silme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete_bank') {
    $id = (int)($_POST['id'] ?? 0);
    
    if ($id) {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM bank_accounts WHERE id = ?");
            $stmt->execute([$id]);
            $success_message = 'Banka hesabı başarıyla silindi.';
        } catch (Exception $e) {
            $error_message = 'Hata: ' . $e->getMessage();
        }
    }
}

// Banka hesaplarını çek
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM bank_accounts ORDER BY created_at DESC");
    $stmt->execute();
    $bank_accounts = $stmt->fetchAll();
} catch (Exception $e) {
    $bank_accounts = [];
    $error_message = 'Veritabanı hatası: ' . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Banka Hesapları - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
</head>
<body class="bg-gray-50">
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <div class="w-64 bg-white shadow-lg">
            <div class="p-6">
                <h2 class="text-xl font-bold text-gray-800">Admin Panel</h2>
                <p class="text-sm text-gray-600">admin@kriptocum.com</p>
            </div>
            <nav class="mt-6">
                <a href="index.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z"></path>
                    </svg>
                    Dashboard
                </a>
                <a href="users.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"></path>
                    </svg>
                    Kullanıcılar
                </a>
                <a href="coins.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Coinler
                </a>
                <a href="bank_accounts.php" class="flex items-center px-6 py-3 bg-blue-50 text-blue-600 border-r-2 border-blue-600">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 0 0-2-2H7a2 2 0 0 0-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 0 1 1-1h2a1 1 0 0 1 1 1v5m-4 0h4"></path>
                    </svg>
                    Banka Hesapları
                </a>
                <a href="crypto_addresses.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Kripto Adresleri
                </a>
                <a href="logout.php" class="flex items-center px-6 py-3 text-red-600 hover:bg-red-50">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path>
                    </svg>
                    Çıkış Yap
                </a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="flex-1 p-8">
            <!-- Header -->
            <div class="mb-8">
                <div class="flex items-center justify-between">
                    <div>
                        <h1 class="text-3xl font-bold text-gray-800">Banka Hesapları</h1>
                        <p class="text-gray-600 mt-2">Para yatırma için banka hesaplarını yönetin</p>
                    </div>
                    <button onclick="openAddModal()" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                        + Yeni Banka Hesabı
                    </button>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if ($success_message): ?>
                <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                    <?php echo $success_message; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <!-- Banka Hesapları Tablosu -->
            <div class="bg-white rounded-lg shadow">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Kayıtlı Banka Hesapları</h3>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Banka</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Hesap Adı</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">IBAN</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Hesap No</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Para Birimi</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Durum</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($bank_accounts as $account): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                        <?php echo htmlspecialchars($account['bank_name']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($account['account_name']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php echo htmlspecialchars($account['iban']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($account['account_number']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($account['currency']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $account['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                            <?php echo $account['is_active'] ? 'Aktif' : 'Pasif'; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <button onclick="openEditModal(<?php echo htmlspecialchars(json_encode($account)); ?>)" class="text-blue-600 hover:text-blue-900 mr-3">
                                            Düzenle
                                        </button>
                                        <button onclick="deleteBank(<?php echo $account['id']; ?>)" class="text-red-600 hover:text-red-900">
                                            Sil
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div id="bankModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                <form method="POST" id="bankForm">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900" id="modalTitle">Yeni Banka Hesabı</h3>
                    </div>
                    <div class="px-6 py-4 space-y-4">
                        <input type="hidden" name="action" id="formAction" value="add_bank">
                        <input type="hidden" name="id" id="bankId">
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Banka Adı</label>
                            <input type="text" name="bank_name" id="bankName" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Hesap Adı</label>
                            <input type="text" name="account_name" id="accountName" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">IBAN</label>
                            <input type="text" name="iban" id="iban" required
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Hesap Numarası</label>
                            <input type="text" name="account_number" id="accountNumber"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Şube Kodu</label>
                            <input type="text" name="branch_code" id="branchCode"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Para Birimi</label>
                            <select name="currency" id="currency" required
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="TRY">TRY - Türk Lirası</option>
                                <option value="USD">USD - Amerikan Doları</option>
                                <option value="EUR">EUR - Euro</option>
                            </select>
                        </div>
                        
                        <div id="activeField" class="hidden">
                            <label class="flex items-center">
                                <input type="checkbox" name="is_active" id="isActive" class="mr-2">
                                <span class="text-sm text-gray-700">Aktif</span>
                            </label>
                        </div>
                    </div>
                    <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                        <button type="button" onclick="closeModal()" class="px-4 py-2 text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300">
                            İptal
                        </button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            Kaydet
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                <div class="px-6 py-4">
                    <h3 class="text-lg font-medium text-gray-900 mb-2">Banka Hesabını Sil</h3>
                    <p class="text-gray-600">Bu banka hesabını silmek istediğinizden emin misiniz? Bu işlem geri alınamaz.</p>
                </div>
                <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                    <button onclick="closeDeleteModal()" class="px-4 py-2 text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300">
                        İptal
                    </button>
                    <form method="POST" id="deleteForm" class="inline">
                        <input type="hidden" name="action" value="delete_bank">
                        <input type="hidden" name="id" id="deleteId">
                        <button type="submit" class="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700">
                            Sil
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script>
        function openAddModal() {
            document.getElementById('modalTitle').textContent = 'Yeni Banka Hesabı';
            document.getElementById('formAction').value = 'add_bank';
            document.getElementById('bankForm').reset();
            document.getElementById('activeField').classList.add('hidden');
            document.getElementById('bankModal').classList.remove('hidden');
        }

        function openEditModal(account) {
            document.getElementById('modalTitle').textContent = 'Banka Hesabını Düzenle';
            document.getElementById('formAction').value = 'update_bank';
            document.getElementById('bankId').value = account.id;
            document.getElementById('bankName').value = account.bank_name;
            document.getElementById('accountName').value = account.account_name;
            document.getElementById('iban').value = account.iban;
            document.getElementById('accountNumber').value = account.account_number || '';
            document.getElementById('branchCode').value = account.branch_code || '';
            document.getElementById('currency').value = account.currency;
            document.getElementById('isActive').checked = account.is_active == 1;
            document.getElementById('activeField').classList.remove('hidden');
            document.getElementById('bankModal').classList.remove('hidden');
        }

        function closeModal() {
            document.getElementById('bankModal').classList.add('hidden');
        }

        function deleteBank(id) {
            document.getElementById('deleteId').value = id;
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.add('hidden');
        }

        // Modal dışına tıklayınca kapat
        window.onclick = function(event) {
            const bankModal = document.getElementById('bankModal');
            const deleteModal = document.getElementById('deleteModal');
            if (event.target === bankModal) {
                closeModal();
            }
            if (event.target === deleteModal) {
                closeDeleteModal();
            }
        }
    </script>
</body>
</html>

