<?php
session_start();
require_once '../config/database.php';

// Admin kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$success_message = '';
$error_message = '';

// Para yatırma talebini onaylama
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'approve_deposit') {
    $deposit_id = (int)($_POST['deposit_id'] ?? 0);
    
    if ($deposit_id) {
        try {
            $db = getDB();
            $db->beginTransaction();
            
            // Deposit bilgilerini al
            $stmt = $db->prepare("SELECT * FROM deposits WHERE id = ?");
            $stmt->execute([$deposit_id]);
            $deposit = $stmt->fetch();
            
            if ($deposit && $deposit['status'] === 'pending') {
                // Kullanıcının cüzdanını kontrol et ve güncelle/oluştur
                $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin_symbol = ?");
                $stmt->execute([$deposit['user_id'], $deposit['coin_symbol']]);
                $wallet = $stmt->fetch();
                
                if ($wallet) {
                    // Mevcut cüzdanı güncelle
                    $stmt = $db->prepare("UPDATE wallets SET balance = balance + ?, available = available + ? WHERE id = ?");
                    $stmt->execute([$deposit['amount'], $deposit['amount'], $wallet['id']]);
                } else {
                    // Yeni cüzdan oluştur
                    $stmt = $db->prepare("INSERT INTO wallets (user_id, coin_symbol, balance, available, frozen) VALUES (?, ?, ?, ?, 0)");
                    $stmt->execute([$deposit['user_id'], $deposit['coin_symbol'], $deposit['amount'], $deposit['amount']]);
                }
                
                // Deposit durumunu güncelle
                $stmt = $db->prepare("UPDATE deposits SET status = 'completed' WHERE id = ?");
                $stmt->execute([$deposit_id]);
                
                // İşlem kaydı oluştur
                $stmt = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'deposit', ?, ?, ?, 'completed', NOW())");
                $stmt->execute([$deposit['user_id'], $deposit['coin_symbol'], $deposit['amount'], $deposit['usd_value']]);
                
                $db->commit();
                $success_message = 'Para yatırma talebi onaylandı ve bakiye eklendi.';
            } else {
                $error_message = 'Deposit bulunamadı veya zaten işlenmiş.';
            }
            
        } catch (Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            $error_message = 'Hata: ' . $e->getMessage();
        }
    }
}

// Para yatırma talebini reddetme
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'reject_deposit') {
    $deposit_id = (int)($_POST['deposit_id'] ?? 0);
    $notes = trim($_POST['notes'] ?? '');
    
    if ($deposit_id) {
        try {
            $db = getDB();
            $stmt = $db->prepare("UPDATE deposits SET status = 'rejected', notes = ? WHERE id = ?");
            $stmt->execute([$notes, $deposit_id]);
            $success_message = 'Para yatırma talebi reddedildi.';
        } catch (Exception $e) {
            $error_message = 'Hata: ' . $e->getMessage();
        }
    }
}

// Para yatırma taleplerini çek
try {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT d.*, u.first_name, u.last_name, u.email 
        FROM deposits d 
        JOIN users u ON d.user_id = u.id 
        ORDER BY d.created_at DESC
    ");
    $stmt->execute();
    $deposits = $stmt->fetchAll();
} catch (Exception $e) {
    $deposits = [];
    $error_message = 'Veritabanı hatası: ' . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Para Yatırma Talepleri - Admin Panel</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
</head>
<body class="bg-gray-50">
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <div class="w-64 bg-white shadow-lg">
            <div class="p-6">
                <h2 class="text-xl font-bold text-gray-800">Admin Panel</h2>
                <p class="text-sm text-gray-600">admin@kriptocum.com</p>
            </div>
            <nav class="mt-6">
                <a href="index.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2H5a2 2 0 00-2-2z"></path>
                    </svg>
                    Dashboard
                </a>
                <a href="users.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"></path>
                    </svg>
                    Kullanıcılar
                </a>
                <a href="coins.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Coinler
                </a>
                <a href="bank_accounts.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 0 0-2-2H7a2 2 0 0 0-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 0 1 1-1h2a1 1 0 0 1 1 1v5m-4 0h4"></path>
                    </svg>
                    Banka Hesapları
                </a>
                <a href="crypto_addresses.php" class="flex items-center px-6 py-3 text-gray-600 hover:bg-gray-100">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Kripto Adresleri
                </a>
                <a href="deposits.php" class="flex items-center px-6 py-3 bg-blue-50 text-blue-600 border-r-2 border-blue-600">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                    Para Yatırma
                </a>
                <a href="logout.php" class="flex items-center px-6 py-3 text-red-600 hover:bg-red-50">
                    <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path>
                    </svg>
                    Çıkış Yap
                </a>
            </nav>
        </div>

        <!-- Main Content -->
        <div class="flex-1 p-8">
            <!-- Header -->
            <div class="mb-8">
                <div class="flex items-center justify-between">
                    <div>
                        <h1 class="text-3xl font-bold text-gray-800">Para Yatırma Talepleri</h1>
                        <p class="text-gray-600 mt-2">Kullanıcıların para yatırma taleplerini yönetin</p>
                    </div>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if ($success_message): ?>
                <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                    <?php echo $success_message; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <!-- Deposits Tablosu -->
            <div class="bg-white rounded-lg shadow">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Para Yatırma Talepleri</h3>
                </div>
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kullanıcı</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Yöntem</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Coin</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Miktar</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">USD Değer</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Durum</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tarih</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($deposits as $deposit): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900">
                                            <?php echo htmlspecialchars($deposit['first_name'] . ' ' . $deposit['last_name']); ?>
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            <?php echo htmlspecialchars($deposit['email']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php
                                        $method_names = [
                                            'bank_transfer' => 'Banka Havalesi',
                                            'crypto_transfer' => 'Kripto Transfer',
                                            'credit_card' => 'Kredi Kartı'
                                        ];
                                        echo $method_names[$deposit['method']] ?? $deposit['method'];
                                        ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($deposit['coin_symbol']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-mono">
                                        <?php 
                                        // TRY için 2 ondalık, diğer coinler için 8 ondalık
                                        $decimals = ($deposit['coin_symbol'] === 'TRY') ? 2 : 8;
                                        $formatted_amount = number_format($deposit['amount'], $decimals);
                                        
                                        // Para birimi sembolü ekle
                                        if ($deposit['coin_symbol'] === 'TRY') {
                                            echo '₺' . $formatted_amount;
                                        } else {
                                            echo $formatted_amount . ' ' . $deposit['coin_symbol'];
                                        }
                                        ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        $<?php echo number_format($deposit['usd_value'], 2); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php
                                        $status_colors = [
                                            'pending' => 'bg-yellow-100 text-yellow-800',
                                            'completed' => 'bg-green-100 text-green-800',
                                            'rejected' => 'bg-red-100 text-red-800',
                                            'cancelled' => 'bg-gray-100 text-gray-800'
                                        ];
                                        $status_names = [
                                            'pending' => 'Beklemede',
                                            'completed' => 'Tamamlandı',
                                            'rejected' => 'Reddedildi',
                                            'cancelled' => 'İptal Edildi'
                                        ];
                                        $color = $status_colors[$deposit['status']] ?? 'bg-gray-100 text-gray-800';
                                        $name = $status_names[$deposit['status']] ?? $deposit['status'];
                                        ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $color; ?>">
                                            <?php echo $name; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo date('d.m.Y H:i', strtotime($deposit['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <?php if ($deposit['status'] === 'pending'): ?>
                                            <div class="flex space-x-2">
                                                <form method="POST" class="inline">
                                                    <input type="hidden" name="action" value="approve_deposit">
                                                    <input type="hidden" name="deposit_id" value="<?php echo $deposit['id']; ?>">
                                                    <button type="submit" class="text-green-600 hover:text-green-900" 
                                                            onclick="return confirm('Bu para yatırma talebini onaylamak istediğinizden emin misiniz?')">
                                                        Onayla
                                                    </button>
                                                </form>
                                                <button onclick="openRejectModal(<?php echo $deposit['id']; ?>)" class="text-red-600 hover:text-red-900">
                                                    Reddet
                                                </button>
                                            </div>
                                        <?php else: ?>
                                            <span class="text-gray-400">-</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Reject Modal -->
    <div id="rejectModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
                <form method="POST" id="rejectForm">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h3 class="text-lg font-medium text-gray-900">Para Yatırma Talebini Reddet</h3>
                    </div>
                    <div class="px-6 py-4">
                        <input type="hidden" name="action" value="reject_deposit">
                        <input type="hidden" name="deposit_id" id="rejectDepositId">
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Red Sebebi (Opsiyonel)</label>
                            <textarea name="notes" id="rejectNotes" rows="3"
                                      class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                      placeholder="Red sebebini açıklayın..."></textarea>
                        </div>
                    </div>
                    <div class="px-6 py-4 bg-gray-50 flex justify-end space-x-3">
                        <button type="button" onclick="closeRejectModal()" class="px-4 py-2 text-gray-700 bg-gray-200 rounded-md hover:bg-gray-300">
                            İptal
                        </button>
                        <button type="submit" class="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700">
                            Reddet
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function openRejectModal(depositId) {
            document.getElementById('rejectDepositId').value = depositId;
            document.getElementById('rejectModal').classList.remove('hidden');
        }

        function closeRejectModal() {
            document.getElementById('rejectModal').classList.add('hidden');
        }

        // Modal dışına tıklayınca kapat
        window.onclick = function(event) {
            const rejectModal = document.getElementById('rejectModal');
            if (event.target === rejectModal) {
                closeRejectModal();
            }
        }
    </script>
</body>
</html>
