<?php
session_start();
require_once 'config/database.php';

// Giriş kontrolü
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

// Dönüştürme işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'convert') {
    $from_coin = strtoupper(trim($_POST['from_coin'] ?? ''));
    $to_coin = strtoupper(trim($_POST['to_coin'] ?? ''));
    $amount = (float)($_POST['amount'] ?? 0);
    
    if ($from_coin && $to_coin && $amount > 0 && $from_coin !== $to_coin) {
        try {
            $db = getDB();
            $db->beginTransaction();
            
            // Kullanıcının cüzdanlarını kontrol et
            $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin_symbol = ?");
            $stmt->execute([$user_id, $from_coin]);
            $from_wallet = $stmt->fetch();
            
            if (!$from_wallet || $from_wallet['available'] < $amount) {
                throw new Exception("Yetersiz {$from_coin} bakiyesi!");
            }
            
            // Dönüştürme oranını hesapla (USD tabanlı dinamik oran)
            $rate = 1; // Varsayılan oran
            $from_usd = $crypto_prices[$from_coin]['raw_price'] ?? null;
            $to_usd = $crypto_prices[$to_coin]['raw_price'] ?? null;
            if ($from_usd && $to_usd && $from_usd > 0 && $to_usd > 0) {
                // 1 FROM = (USD_FROM / USD_TO) TO
                $rate = (float)$from_usd / (float)$to_usd;
            } else {
                // Eski sabit fallback'ler
                if ($from_coin === 'TRY' && $to_coin === 'USDT') {
                    $rate = 0.033;
                } elseif ($from_coin === 'USDT' && $to_coin === 'TRY') {
                    $rate = 30.30;
                } elseif ($from_coin === 'TRY' && $to_coin === 'BTC') {
                    $rate = 0.00000029;
                } elseif ($from_coin === 'BTC' && $to_coin === 'TRY') {
                    $rate = 3450000;
                }
            }
            
            $to_amount = $amount * $rate;
            
            // Kaynak cüzdanından düş
            $stmt = $db->prepare("UPDATE wallets SET balance = balance - ?, available = available - ? WHERE id = ?");
            $stmt->execute([$amount, $amount, $from_wallet['id']]);
            
            // Hedef cüzdanı kontrol et ve güncelle/oluştur
            $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND coin_symbol = ?");
            $stmt->execute([$user_id, $to_coin]);
            $to_wallet = $stmt->fetch();
            
            if ($to_wallet) {
                $stmt = $db->prepare("UPDATE wallets SET balance = balance + ?, available = available + ? WHERE id = ?");
                $stmt->execute([$to_amount, $to_amount, $to_wallet['id']]);
            } else {
                $stmt = $db->prepare("INSERT INTO wallets (user_id, coin_symbol, balance, available, frozen) VALUES (?, ?, ?, ?, 0)");
                $stmt->execute([$user_id, $to_coin, $to_amount, $to_amount]);
            }
            
            // Dönüştürme kaydı oluştur
            $stmt = $db->prepare("INSERT INTO conversions (user_id, from_coin, to_coin, from_amount, to_amount, rate, status) VALUES (?, ?, ?, ?, ?, ?, 'completed')");
            $stmt->execute([$user_id, $from_coin, $to_coin, $amount, $to_amount, $rate]);
            
            // İşlem kaydı oluştur
            $stmt = $db->prepare("INSERT INTO transactions (user_id, type, coin_symbol, amount, price, status, created_at) VALUES (?, 'convert', ?, ?, ?, 'completed', NOW())");
            $stmt->execute([$user_id, $from_coin, -$amount, $rate]);
            
            $db->commit();
            $success_message = "Dönüştürme başarılı! {$amount} {$from_coin} → {$to_amount} {$to_coin}";
            
        } catch (Exception $e) {
            if ($db->inTransaction()) {
                $db->rollBack();
            }
            $error_message = 'Hata: ' . $e->getMessage();
        }
    } else {
        $error_message = 'Lütfen geçerli bilgileri girin.';
    }
}

// Kullanıcının cüzdanlarını çek
try {
    $db = getDB();
    // Tüm cüzdanları çek (sıfır bakiyeli olanlar dahil)
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? ORDER BY balance DESC");
    $stmt->execute([$user_id]);
    $all_wallets = $stmt->fetchAll();
    
    // Sadece pozitif bakiyeli olanları göster
    $wallets = array_filter($all_wallets, function($wallet) {
        return $wallet['balance'] > 0;
    });
    
    // TRY ve USDT bakiyelerini ayrı ayrı çek
    $try_balance = 0;
    $usdt_balance = 0;
    
    foreach ($all_wallets as $wallet) {
        if ($wallet['coin_symbol'] === 'TRY') {
            $try_balance = $wallet['available'];
        } elseif ($wallet['coin_symbol'] === 'USDT') {
            $usdt_balance = $wallet['available'];
        }
    }
} catch (Exception $e) {
    $wallets = [];
    $all_wallets = [];
    $try_balance = 0;
    $usdt_balance = 0;
}

// Kripto fiyatlarını yükle
$crypto_prices = [];
if (file_exists('assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices = $price_data['prices'];
    }
}

// Bakiye formatlama fonksiyonu
function formatBalance($amount, $symbol) {
    if ($amount == 0) return '0';
    
    // TRY için özel format
    if ($symbol === 'TRY') {
        return number_format($amount, 2);
    }
    
    // USDT için özel format (kısaltma yok)
    if ($symbol === 'USDT') {
        return number_format($amount, 2);
    }
    
    // Diğer coinler için akıllı format
    if ($amount >= 1000000) {
        return number_format($amount / 1000000, 2) . 'M';
    } else if ($amount >= 1000) {
        return number_format($amount / 1000, 2) . 'K';
    } else if ($amount >= 1) {
        return number_format($amount, 4);
    } else if ($amount >= 0.01) {
        return number_format($amount, 6);
    } else {
        return number_format($amount, 8);
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Para Dönüştürme - KriptoCum</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
</head>
<body class="bg-gray-50">
    <?php include 'components/header.php'; ?>
    
    <div class="min-h-screen bg-gray-50 pt-20 pb-8">
        <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <!-- Header -->
            <div class="mb-8 text-center">
                <div class="inline-flex items-center justify-center w-16 h-16 bg-blue-100 rounded-full mb-4">
                    <svg class="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path>
                    </svg>
                </div>
                <h1 class="text-4xl font-bold text-gray-900 mb-2">Para Dönüştürme Merkezi</h1>
                <p class="text-lg text-gray-600">Paranızı istediğiniz kripto para birimine güvenle dönüştürün</p>
                <div class="mt-4 flex justify-center space-x-6 text-sm text-gray-500">
                    <div class="flex items-center">
                        <svg class="w-4 h-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Anında
                    </div>
                    <div class="flex items-center">
                        <svg class="w-4 h-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Güvenli
                    </div>
                    <div class="flex items-center">
                        <svg class="w-4 h-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Ücretsiz
                    </div>
                </div>
            </div>

            <!-- Mevcut Bakiyeler -->
            <div class="mb-8">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="bg-white rounded-lg p-4 shadow-sm border">
                        <div class="text-2xl font-bold text-blue-600">₺<?php echo formatBalance($try_balance, 'TRY'); ?></div>
                        <div class="text-sm text-gray-500">TL Bakiyeniz</div>
                    </div>
                    <div class="bg-white rounded-lg p-4 shadow-sm border">
                        <div class="text-2xl font-bold text-green-600">$<?php echo formatBalance($usdt_balance, 'USDT'); ?></div>
                        <div class="text-sm text-gray-500">USDT Bakiyeniz</div>
                    </div>
                    <div class="bg-white rounded-lg p-4 shadow-sm border">
                        <div class="text-2xl font-bold text-purple-600"><?php echo count($wallets); ?></div>
                        <div class="text-sm text-gray-500">Aktif Cüzdan</div>
                    </div>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if ($success_message): ?>
                <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                    <?php echo $success_message; ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Dönüştürme Formu -->
                <div class="lg:col-span-2">
                    <div class="bg-white rounded-lg shadow p-6">
                        <h2 class="text-xl font-semibold text-gray-900 mb-6">Para Dönüştürme</h2>
                        
                        <form method="POST" class="space-y-6">
                            <input type="hidden" name="action" value="convert">
                            
                            <!-- Kaynak Para Birimi -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Kaynak Para Birimi</label>
                                <select name="from_coin" id="from_coin" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Kaynak para birimi seçin</option>
                                    <?php foreach ($all_wallets as $wallet): ?>
                                        <option value="<?php echo $wallet['coin_symbol']; ?>" data-balance="<?php echo $wallet['available']; ?>">
                                            <?php echo $wallet['coin_symbol']; ?> - 
                                            <?php echo formatBalance($wallet['available'], $wallet['coin_symbol']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <!-- Hedef Para Birimi -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Hedef Para Birimi</label>
                                <select name="to_coin" id="to_coin" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Hedef para birimi seçin</option>
                                    <option value="TRY">₺ Türk Lirası (TRY)</option>
                                    <option value="USDT">$ Tether (USDT)</option>
                                    <option value="BTC">₿ Bitcoin (BTC)</option>
                                    <option value="ETH">Ξ Ethereum (ETH)</option>
                                    <option value="SOL">◎ Solana (SOL)</option>
                                    <option value="BNB">🟡 Binance Coin (BNB)</option>
                                </select>
                            </div>

                            <!-- Miktar -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Miktar</label>
                                <div class="relative">
                                    <input type="number" name="amount" id="amount" step="0.00000001" min="0" 
                                           class="w-full px-3 py-2 pr-16 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" 
                                           placeholder="0.00" required>
                                    <div class="absolute inset-y-0 right-0 pr-3 flex items-center">
                                        <span id="from_coin_symbol" class="text-gray-500 text-sm">₺</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Dönüştürme Önizlemesi -->
                            <div id="conversion_preview" class="bg-gray-50 rounded-lg p-4 border hidden">
                                <h3 class="text-sm font-medium text-gray-900 mb-2">Dönüştürme Önizlemesi</h3>
                                <div class="space-y-2">
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray-600">Dönüştürülecek:</span>
                                        <span id="preview_from" class="font-semibold text-gray-900"></span>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray-600">Alacağınız:</span>
                                        <span id="preview_to" class="font-bold text-green-600 text-lg"></span>
                                    </div>
                                    <div class="flex justify-between items-center text-sm">
                                        <span class="text-gray-500">Dönüştürme Oranı:</span>
                                        <span id="preview_rate" class="text-gray-700"></span>
                                    </div>
                                </div>
                            </div>

                            <button type="submit" class="w-full bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white py-3 px-4 rounded-md transition-all duration-300 shadow-sm hover:shadow-md font-medium">
                                Dönüştür
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Bilgi Paneli -->
                <div class="space-y-4">
                    <!-- Mevcut Bakiyeler -->
                    <div class="bg-white rounded-lg shadow p-4">
                        <h3 class="text-sm font-semibold text-gray-900 mb-3">Mevcut Bakiyeler</h3>
                        <div class="space-y-2">
                            <?php if (empty($wallets)): ?>
                                <div class="text-center py-4">
                                    <p class="text-gray-500 text-xs">Henüz cüzdanınızda para yok.</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($wallets as $wallet): ?>
                                    <div class="flex justify-between items-center p-2 bg-gray-50 rounded text-xs">
                                        <div class="flex items-center space-x-2">
                                            <div class="w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center">
                                                <span class="text-white text-xs font-bold"><?php echo strtoupper(substr($wallet['coin_symbol'], 0, 2)); ?></span>
                                            </div>
                                            <span class="font-medium"><?php echo $wallet['coin_symbol']; ?></span>
                                        </div>
                                        <span class="font-semibold"><?php echo formatBalance($wallet['available'], $wallet['coin_symbol']); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Dönüştürme Oranları -->
                    <div class="bg-white rounded-lg shadow p-4">
                        <h3 class="text-sm font-semibold text-gray-900 mb-3">Güncel Oranlar</h3>
                        <div class="space-y-2 text-xs" id="dynamic_rates">
                            <!-- JS ile güncellenecek -->
                        </div>
                    </div>

                    <!-- Bilgi Notları -->
                    <div class="bg-white rounded-lg shadow p-4">
                        <h3 class="text-sm font-semibold text-gray-900 mb-3">Önemli Bilgiler</h3>
                        <div class="space-y-1 text-xs text-gray-600">
                            <div>• Dönüştürme işlemi anında gerçekleşir</div>
                            <div>• İşlem ücreti alınmaz</div>
                            <div>• Oranlar güncel piyasa fiyatlarına göre hesaplanır</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Dinamik fiyatlar (PHP'den)
        const prices = <?php echo json_encode($crypto_prices ?: new stdClass()); ?>;
        
        // Debug: Fiyatları konsola yazdır
        console.log('Yüklenen fiyatlar:', prices);
        console.log('TRY fiyatı:', prices.TRY?.raw_price);
        console.log('USDT fiyatı:', prices.USDT?.raw_price);
        
        // USDT için varsayılan değer (1 USDT = 1 USD)
        if (!prices.USDT || !prices.USDT.raw_price) {
            prices.USDT = { raw_price: 1 };
            console.log('USDT için varsayılan değer kullanıldı: 1');
        }
        
        function getRate(from, to) {
            let f = prices[from]?.raw_price;
            let t = prices[to]?.raw_price;
            
            // USDT için varsayılan
            if (to === 'USDT' && (!t || parseFloat(t) <= 0)) {
                t = 1;
            }
            if (from === 'USDT' && (!f || parseFloat(f) <= 0)) {
                f = 1;
            }
            
            // TRY için fallback (1 USD = 42.05 TRY, yani 1 TRY = 1/42.05 USD)
            if (from === 'TRY' && (!f || parseFloat(f) <= 0)) {
                f = 1.0 / 42.05; // ~0.02378
            }
            if (to === 'TRY' && (!t || parseFloat(t) <= 0)) {
                t = 1.0 / 42.05; // ~0.02378
            }
            
            if (f && t && parseFloat(f) > 0 && parseFloat(t) > 0) {
                const rate = parseFloat(f) / parseFloat(t);
                return rate;
            }
            
            console.warn(`Oran hesaplanamadı: ${from} -> ${to}`, { f, t, prices });
            return 1; // Fallback
        }

        // Kaynak para birimi değiştiğinde
        document.getElementById('from_coin').addEventListener('change', function() {
            updateCurrencySymbol();
            updateBalanceInfo();
            updateConversionPreview();
        });

        // Hedef para birimi değiştiğinde
        document.getElementById('to_coin').addEventListener('change', function() {
            updateConversionPreview();
        });

        // Miktar değiştiğinde
        document.getElementById('amount').addEventListener('input', function() {
            updateConversionPreview();
        });

        // Maksimum kullan butonu
        document.getElementById('use_max').addEventListener('click', function() {
            const fromCoin = document.getElementById('from_coin').value;
            const selectedOption = document.querySelector(`#from_coin option[value="${fromCoin}"]`);
            if (selectedOption) {
                const balance = parseFloat(selectedOption.dataset.balance);
                document.getElementById('amount').value = balance;
                updateConversionPreview();
            }
        });

        // Swap currencies butonu
        document.getElementById('swap_currencies').addEventListener('click', function() {
            const fromCoin = document.getElementById('from_coin').value;
            const toCoin = document.getElementById('to_coin').value;
            
            if (fromCoin && toCoin) {
                document.getElementById('from_coin').value = toCoin;
                document.getElementById('to_coin').value = fromCoin;
                updateCurrencySymbol();
                updateBalanceInfo();
                updateConversionPreview();
            }
        });

        function updateCurrencySymbol() {
            const fromCoin = document.getElementById('from_coin').value;
            const symbols = {
                'TRY': '₺',
                'USDT': '$',
                'BTC': '₿',
                'ETH': 'Ξ',
                'SOL': '◎',
                'BNB': 'BNB'
            };
            document.getElementById('from_coin_symbol').textContent = symbols[fromCoin] || '₺';
        }

        function updateBalanceInfo() {
            const fromCoin = document.getElementById('from_coin').value;
            const selectedOption = document.querySelector(`#from_coin option[value="${fromCoin}"]`);
            const balanceInfo = document.getElementById('from_balance_info');
            const balanceAmount = document.getElementById('from_balance_amount');
            
            if (selectedOption && fromCoin) {
                const balance = parseFloat(selectedOption.dataset.balance);
                balanceAmount.textContent = formatBalance(balance, fromCoin) + ' ' + fromCoin;
                balanceInfo.classList.remove('hidden');
            } else {
                balanceInfo.classList.add('hidden');
            }
        }

        function formatBalance(amount, symbol) {
            if (amount === 0) return '0';
            
            // TRY için özel format
            if (symbol === 'TRY') {
                return amount.toFixed(2);
            }
            
            // USDT için özel format (kısaltma yok)
            if (symbol === 'USDT') {
                return amount.toFixed(2);
            }
            
            // Diğer coinler için akıllı format
            if (amount >= 1000000) {
                return (amount / 1000000).toFixed(2) + 'M';
            } else if (amount >= 1000) {
                return (amount / 1000).toFixed(2) + 'K';
            } else if (amount >= 1) {
                return amount.toFixed(4);
            } else if (amount >= 0.01) {
                return amount.toFixed(6);
            } else {
                return amount.toFixed(8);
            }
        }

        function updateConversionPreview() {
            const fromCoin = document.getElementById('from_coin').value;
            const toCoin = document.getElementById('to_coin').value;
            const amount = parseFloat(document.getElementById('amount').value) || 0;
            
            const preview = document.getElementById('conversion_preview');
            const previewFrom = document.getElementById('preview_from');
            const previewTo = document.getElementById('preview_to');
            const previewRate = document.getElementById('preview_rate');
            
            if (fromCoin && toCoin && amount > 0) {
                const rate = getRate(fromCoin, toCoin);
                const convertedAmount = amount * rate;
                
                const symbols = {
                    'TRY': '₺',
                    'USDT': '$',
                    'BTC': '₿',
                    'ETH': 'Ξ',
                    'SOL': '◎',
                    'BNB': 'BNB'
                };
                
                previewFrom.textContent = `${formatBalance(amount, fromCoin)} ${fromCoin}`;
                previewTo.textContent = `${formatBalance(convertedAmount, toCoin)} ${toCoin}`;
                
                // Oran formatı: küçük sayılar için daha fazla ondalık
                let rateDisplay = rate;
                if (rate < 0.01) {
                    rateDisplay = rate.toFixed(8);
                } else if (rate < 1) {
                    rateDisplay = rate.toFixed(6);
                } else if (rate < 100) {
                    rateDisplay = rate.toFixed(4);
                } else {
                    rateDisplay = rate.toFixed(2);
                }
                previewRate.textContent = `1 ${fromCoin} = ${rateDisplay} ${toCoin}`;
                
                preview.classList.remove('hidden');
            } else {
                preview.classList.add('hidden');
            }
        }

        // Dinamik oran kutusunu doldur
        function renderDynamicRates() {
            const box = document.getElementById('dynamic_rates');
            if (!box) return;
            const pairs = [
                ['TRY','USDT'],
                ['USDT','TRY'],
                ['TRY','BTC'],
                ['BTC','TRY']
            ];
            box.innerHTML = pairs.map(([a,b]) => {
                const r = getRate(a,b);
                let rateDisplay = r;
                if (r < 0.01) {
                    rateDisplay = r.toFixed(8);
                } else if (r < 1) {
                    rateDisplay = r.toFixed(6);
                } else if (r < 100) {
                    rateDisplay = r.toFixed(4);
                } else {
                    rateDisplay = r.toFixed(2);
                }
                return `<div class=\"flex justify-between\"><span class=\"text-gray-600\">1 ${a} =</span><span class=\"font-medium\">${rateDisplay} ${b}</span></div>`;
            }).join('');
        }

        document.addEventListener('DOMContentLoaded', () => {
            renderDynamicRates();
        });
    </script>
</body>
</html>
