<?php
session_start();
require_once 'config/database.php';

// Giriş kontrolü
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

// Para yatırma işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'deposit') {
    $method = $_POST['method'] ?? '';
    $coin_symbol = strtoupper(trim($_POST['coin_symbol'] ?? ''));
    $amount = (float)($_POST['amount'] ?? 0);
    $bank_account = $_POST['bank_account'] ?? '';
    $crypto_address = $_POST['crypto_address'] ?? '';
    
    if ($coin_symbol && $amount > 0) {
        try {
            $db = getDB();
            
            // USD değerini hesapla
            $usd_value = $amount;
            if ($coin_symbol === 'TRY') {
                $usd_value = $amount * 0.033; // 1 TRY = 0.033 USD
            }
            
            // Deposit kaydı oluştur
            $stmt = $db->prepare("
                INSERT INTO deposits (user_id, method, coin_symbol, amount, usd_value, bank_account, crypto_address, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')
            ");
            $stmt->execute([$user_id, $method, $coin_symbol, $amount, $usd_value, $bank_account, $crypto_address]);
            
            $success_message = 'Para yatırma talebiniz başarıyla alındı! Onay süresi: 5-30 dakika. Onaylandığında bakiye otomatik olarak hesabınıza eklenecektir.';
            
        } catch (Exception $e) {
            $error_message = 'Hata: ' . $e->getMessage();
        }
    } else {
        $error_message = 'Lütfen geçerli bilgileri girin.';
    }
}

// Kripto fiyatlarını yükle
$crypto_prices = [];
if (file_exists('assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices = $price_data['prices'];
    }
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Para Yatırma - KriptoCum</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <?php include 'components/header.php'; ?>
    
    <div class="min-h-screen bg-gray-50 py-8 pt-20">
        <div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <!-- Header -->
            <div class="mb-8 text-center">
                <h1 class="text-4xl font-bold text-gray-900 mb-2">Para Yatırma Merkezi</h1>
                <p class="text-lg text-gray-600">Hesabınıza güvenli ve hızlı para yatırın</p>
                <div class="mt-4 flex justify-center space-x-6 text-sm text-gray-500">
                    <div class="flex items-center">
                        <svg class="w-4 h-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Güvenli
                    </div>
                    <div class="flex items-center">
                        <svg class="w-4 h-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                        </svg>
                        Hızlı
                    </div>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if ($success_message): ?>
                <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded-lg">
                    <div class="flex items-center">
                        <svg class="w-5 h-5 mr-2 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        <div>
                            <p class="font-medium">✅ Başarılı!</p>
                            <p class="text-sm mt-1"><?php echo $success_message; ?></p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="mb-6 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Yatırma Formu -->
                <div class="lg:col-span-2">
                    <div class="bg-white rounded-lg shadow p-6">
                        <h2 class="text-xl font-semibold text-gray-900 mb-6">Para Yatırma</h2>
                        
                        <form method="POST" class="space-y-6">
                            <input type="hidden" name="action" value="deposit">
                            
                            <!-- Yatırma Yöntemi -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-3">Yatırma Yöntemi</label>
                                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    <label class="relative cursor-pointer">
                                        <input type="radio" name="method" value="bank_transfer" class="sr-only" required>
                                        <div class="border-2 border-gray-200 rounded-lg p-4 text-center hover:border-blue-500 transition-colors deposit-method">
                                            <div class="text-2xl mb-2">🏦</div>
                                            <div class="font-medium">Banka Havalesi</div>
                                            <div class="text-sm text-gray-500">Türk Lirası</div>
                                        </div>
                                    </label>
                                    
                                    <label class="relative cursor-pointer">
                                        <input type="radio" name="method" value="crypto_transfer" class="sr-only" required>
                                        <div class="border-2 border-gray-200 rounded-lg p-4 text-center hover:border-blue-500 transition-colors deposit-method">
                                            <div class="text-2xl mb-2">₿</div>
                                            <div class="font-medium">Kripto Transfer</div>
                                            <div class="text-sm text-gray-500">BTC, ETH, USDT</div>
                                        </div>
                                    </label>
                                    
                                    <label class="relative cursor-pointer">
                                        <input type="radio" name="method" value="credit_card" class="sr-only" required>
                                        <div class="border-2 border-gray-200 rounded-lg p-4 text-center hover:border-blue-500 transition-colors deposit-method">
                                            <div class="text-2xl mb-2">💳</div>
                                            <div class="font-medium">Kredi Kartı</div>
                                            <div class="text-sm text-gray-500">Hızlı Yatırma</div>
                                        </div>
                                    </label>
                                </div>
                            </div>

                            <!-- Coin Seçimi -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Para Birimi</label>
                                <select name="coin_symbol" id="coin_symbol" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" required>
                                    <option value="">Para birimi seçin</option>
                                    <option value="TRY">₺ Türk Lirası (TRY)</option>
                                    <option value="USDT">$ Tether (USDT)</option>
                                    <option value="BTC">₿ Bitcoin (BTC)</option>
                                    <option value="ETH">Ξ Ethereum (ETH)</option>
                                </select>
                            </div>

                            <!-- Miktar -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Miktar</label>
                                <div class="relative">
                                    <input type="number" name="amount" step="0.00000001" min="0" 
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" 
                                           placeholder="0.00" required>
                                    <div class="absolute inset-y-0 right-0 pr-3 flex items-center">
                                        <span class="text-gray-500 text-sm" id="currency_symbol">₺</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Banka Hesabı (Banka Havalesi için) -->
                            <div id="bank_account_section" class="hidden">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Banka Hesap Bilgileri</label>
                                <select name="bank_account" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Banka seçin</option>
                                    <?php
                                    // Veritabanından banka hesaplarını çek
                                    try {
                                        $db = getDB();
                                        $stmt = $db->prepare("SELECT * FROM bank_accounts WHERE is_active = 1 ORDER BY bank_name");
                                        $stmt->execute();
                                        $bank_accounts = $stmt->fetchAll();
                                        
                                        if (count($bank_accounts) > 0) {
                                            foreach ($bank_accounts as $bank) {
                                                echo '<option value="' . htmlspecialchars($bank['id']) . '">';
                                                echo htmlspecialchars($bank['bank_name']) . ' - ' . htmlspecialchars($bank['account_name']) . ' (' . htmlspecialchars($bank['iban']) . ')';
                                                echo '</option>';
                                            }
                                        } else {
                                            // Hata durumunda varsayılan seçenekler
                                            echo '<option value="ziraat">Ziraat Bankası - TRY</option>';
                                            echo '<option value="isbank">İş Bankası - TRY</option>';
                                            echo '<option value="garanti">Garanti BBVA - TRY</option>';
                                        }
                                    } catch (Exception $e) {
                                        // Hata durumunda varsayılan seçenekler
                                        echo '<option value="ziraat">Ziraat Bankası - TRY</option>';
                                        echo '<option value="isbank">İş Bankası - TRY</option>';
                                        echo '<option value="garanti">Garanti BBVA - TRY</option>';
                                    }
                                    ?>
                                </select>
                            </div>

                            <!-- Kripto Adresi (Kripto Transfer için) -->
                            <div id="crypto_address_section" class="hidden">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Kripto Adresi</label>
                                <select name="crypto_address" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Kripto adresi seçin</option>
                                    <?php
                                    // Veritabanından kripto adreslerini çek
                                    try {
                                        $db = getDB();
                                        $stmt = $db->prepare("SELECT * FROM crypto_addresses WHERE is_active = 1 ORDER BY coin_symbol");
                                        $stmt->execute();
                                        $crypto_addresses = $stmt->fetchAll();
                                        
                                        foreach ($crypto_addresses as $crypto) {
                                            echo '<option value="' . htmlspecialchars($crypto['id']) . '">';
                                            echo htmlspecialchars($crypto['coin_symbol']) . ' - ' . htmlspecialchars($crypto['address']);
                                            if ($crypto['network']) {
                                                echo ' (' . htmlspecialchars($crypto['network']) . ')';
                                            }
                                            echo '</option>';
                                        }
                                    } catch (Exception $e) {
                                        // Hata durumunda varsayılan seçenekler
                                        echo '<option value="btc">BTC - Bitcoin Adresi</option>';
                                        echo '<option value="eth">ETH - Ethereum Adresi</option>';
                                        echo '<option value="usdt">USDT - Tether Adresi</option>';
                                    }
                                    ?>
                                </select>
                            </div>

                            <button type="submit" class="w-full bg-gradient-to-r from-gray-700 to-gray-600 hover:from-gray-600 hover:to-gray-500 text-white py-3 px-4 rounded-md transition-all duration-300 shadow-sm hover:shadow-md font-medium">
                                Para Yatırma Talebi Gönder
                            </button>
                        </form>
                    </div>
                </div>

                <!-- Bilgi Paneli -->
                <div class="space-y-6">
                    <!-- Güncel Fiyatlar -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Güncel Fiyatlar</h3>
                        <div class="space-y-3">
                            <?php foreach (['TRY', 'USDT', 'BTC', 'ETH'] as $symbol): ?>
                                <?php if (isset($crypto_prices[$symbol])): ?>
                                    <div class="flex justify-between items-center">
                                        <div class="flex items-center space-x-2">
                                            <span class="text-sm font-medium"><?php echo $symbol; ?></span>
                                        </div>
                                        <div class="text-right">
                                            <div class="text-sm font-semibold"><?php echo $crypto_prices[$symbol]['price'] ?? 'N/A'; ?></div>
                                            <div class="text-xs text-gray-500"><?php echo $crypto_prices[$symbol]['change'] ?? 'N/A'; ?></div>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <!-- Yatırma Bilgileri -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">Yatırma Bilgileri</h3>
                        <div class="space-y-3 text-sm text-gray-600">
                            <div class="flex justify-between">
                                <span>Minimum Yatırma:</span>
                                <span class="font-medium">₺50</span>
                            </div>
                            <div class="flex justify-between">
                                <span>İşlem Ücreti:</span>
                                <span class="font-medium text-green-600">Ücretsiz</span>
                            </div>
                            <div class="flex justify-between">
                                <span>Onay Süresi:</span>
                                <span class="font-medium">5-30 dakika</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Yatırma yöntemi değiştiğinde
        document.querySelectorAll('input[name="method"]').forEach(radio => {
            radio.addEventListener('change', function() {
                const bankSection = document.getElementById('bank_account_section');
                const cryptoSection = document.getElementById('crypto_address_section');
                
                bankSection.classList.add('hidden');
                cryptoSection.classList.add('hidden');
                
                if (this.value === 'bank_transfer') {
                    bankSection.classList.remove('hidden');
                } else if (this.value === 'crypto_transfer') {
                    cryptoSection.classList.remove('hidden');
                }
            });
        });

        // Coin değiştiğinde para birimi sembolünü güncelle
        document.getElementById('coin_symbol').addEventListener('change', function() {
            const currencySymbol = document.getElementById('currency_symbol');
            const symbols = {
                'TRY': '₺',
                'USDT': '$',
                'BTC': '₿',
                'ETH': 'Ξ'
            };
            currencySymbol.textContent = symbols[this.value] || '₺';
        });

        // Yatırma yöntemi seçimini görsel olarak göster
        document.querySelectorAll('.deposit-method').forEach(method => {
            method.addEventListener('click', function() {
                // Tüm method'ları temizle
                document.querySelectorAll('.deposit-method').forEach(m => {
                    m.classList.remove('border-blue-500', 'bg-blue-50');
                    m.classList.add('border-gray-200');
                });
                
                // Seçili method'u işaretle
                this.classList.remove('border-gray-200');
                this.classList.add('border-blue-500', 'bg-blue-50');
            });
        });
    </script>
</body>
</html>
