<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];

// Kullanıcının cüzdan verilerini veritabanından çek
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
} catch (Exception $e) {
    $wallets = [];
}

// Kripto para fiyatlarını yükle
$crypto_prices = [];
if (file_exists('assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices = $price_data['prices'];
    }
}

// Toplam bakiyeleri hesapla
$total_balance = 0;
$available_balance = 0;
$in_orders = 0;
$active_wallets = 0; // Bakiyesi 0'dan büyük olan cüzdan sayısı

foreach ($wallets as $wallet) {
    $coin_symbol = $wallet['coin_symbol'];
    $price = 1; // Default price
    
    // TRY'yi toplam bakiyeden hariç tut (ayrı gösterilecek)
    if ($coin_symbol === 'TRY') {
        // Aktif varlık sayısını say (TRY dahil)
        if ($wallet['balance'] > 0) {
            $active_wallets++;
        }
        continue; // TRY'yi toplam bakiyeden atla
    }
    
    if (isset($crypto_prices[$coin_symbol])) {
        $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
    }
    
    $total_balance += $wallet['balance'] * $price;
    $available_balance += $wallet['available'] * $price;
    $in_orders += $wallet['frozen'] * $price;
    
    // Aktif varlık sayısını say
    if ($wallet['balance'] > 0) {
        $active_wallets++;
    }
}

// Son işlemleri çek (transactions ve conversions birleştir)
try {
    // Transactions tablosundan işlemleri çek
    $stmt = $db->prepare("SELECT *, 'transaction' as source FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 10");
    $stmt->execute([$user_id]);
    $transactions = $stmt->fetchAll();
    
    // Conversions tablosundan dönüştürme işlemlerini çek
    $stmt = $db->prepare("SELECT *, 'conversion' as source FROM conversions WHERE user_id = ? ORDER BY created_at DESC LIMIT 10");
    $stmt->execute([$user_id]);
    $conversions = $stmt->fetchAll();
    
    // Her iki tabloyu birleştir ve tarihe göre sırala
    $all_transactions = array_merge($transactions, $conversions);
    usort($all_transactions, function($a, $b) {
        return strtotime($b['created_at']) - strtotime($a['created_at']);
    });
    
    $recent_transactions = array_slice($all_transactions, 0, 10);
} catch (Exception $e) {
    $recent_transactions = [];
}

include 'components/header.php';
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Portföy - Kriptocum</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
    <!-- Chart.js for graphs -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <!-- Real-time updates -->
    <script src="https://cdn.jsdelivr.net/npm/socket.io-client@4.7.2/dist/socket.io.min.js"></script>
    <style>
        .number-animation {
            animation: numberFlash 6s ease-in-out infinite;
        }
        
        @keyframes numberFlash {
            0% { color: #dc2626; }  /* Kırmızı */
            50% { color: #dc2626; } /* Kırmızı (3 saniye) */
            50.1% { color: #000000; } /* Siyah */
            100% { color: #000000; } /* Siyah (3 saniye) */
        }
        
        /* Grafik konteynerleri için sabit yükseklik */
        .chart-container {
            height: 300px !important;
            max-height: 300px !important;
            overflow: hidden;
        }
        
        /* Canvas elementleri için boyut sınırlaması */
        canvas {
            max-height: 300px !important;
            width: 100% !important;
            height: 100% !important;
        }
    </style>
</head>
<body class="bg-gray-50">
    
    <main class="pt-16">
        <div class="container mx-auto px-4 py-6">
            <!-- Portfolio Header -->
            <div class="mb-6 md:mb-8">
                <h1 class="text-2xl md:text-3xl font-bold text-gray-800 mb-1 md:mb-2">Portföy</h1>
                <p class="text-sm md:text-base text-gray-600">Kripto para varlıklarınızı yönetin</p>
            </div>
            
            <!-- Balance Overview -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 md:gap-6 mb-6 md:mb-8">
                <div class="bg-white rounded-xl shadow-sm p-4 md:p-6 border border-gray-200">
                    <div class="text-xs md:text-sm text-gray-600 mb-1 md:mb-2">Toplam Bakiye</div>
                    <div class="text-2xl md:text-3xl font-bold text-gray-800 number-animation">$<?php echo number_format($total_balance, 2); ?></div>
                    <div class="text-xs md:text-sm text-green-600 mt-1 md:mt-2">Gerçek zamanlı</div>
                </div>
                <div class="bg-white rounded-xl shadow-sm p-4 md:p-6 border border-gray-200">
                    <div class="text-xs md:text-sm text-gray-600 mb-1 md:mb-2">Müsait</div>
                    <div class="text-2xl md:text-3xl font-bold text-gray-800 number-animation">$<?php echo number_format($available_balance, 2); ?></div>
                    <div class="text-xs md:text-sm text-gray-500 mt-1 md:mt-2">İşlem için hazır</div>
                </div>
                <div class="bg-white rounded-xl shadow-sm p-4 md:p-6 border border-gray-200">
                    <div class="text-xs md:text-sm text-gray-600 mb-1 md:mb-2">Emirlerde</div>
                    <div class="text-2xl md:text-3xl font-bold text-gray-800 number-animation">$<?php echo number_format($in_orders, 2); ?></div>
                    <div class="text-xs md:text-sm text-gray-500 mt-1 md:mt-2">Beklemede</div>
                </div>
                <!-- TRY Balance Card -->
                <?php
                $try_balance = 0;
                foreach ($wallets as $wallet) {
                    if ($wallet['coin_symbol'] === 'TRY') {
                        $try_balance = $wallet['balance'];
                        break;
                    }
                }
                ?>
                <div class="bg-gradient-to-r from-red-50 to-red-100 rounded-xl shadow-sm p-4 md:p-6 border border-red-200">
                    <div class="flex items-center justify-between">
                        <div>
                            <div class="text-xs md:text-sm text-gray-600 mb-1 md:mb-2">TL Bakiye</div>
                            <div class="text-2xl md:text-3xl font-bold text-gray-800 number-animation">₺<?php echo number_format($try_balance, 2); ?></div>
                            <div class="text-xs md:text-sm text-red-600 mt-1 md:mt-2">Türk Lirası</div>
                        </div>
                        <div class="p-2.5 md:p-3 bg-red-500 rounded-full">
                            <span class="text-xl md:text-2xl font-bold text-white">₺</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Charts and Analytics Section -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                <!-- Portfolio Performance Chart -->
                <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-200">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="text-lg font-semibold text-gray-800">Portföy Performansı</h3>
                        <div class="flex gap-2">
                            <button onclick="updateChart('7d')" class="px-3 py-1 text-xs bg-blue-100 text-blue-600 rounded-full hover:bg-blue-200">7G</button>
                            <button onclick="updateChart('30d')" class="px-3 py-1 text-xs bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200">30G</button>
                            <button onclick="updateChart('90d')" class="px-3 py-1 text-xs bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200">90G</button>
                        </div>
                    </div>
                    <div class="relative chart-container">
                        <canvas id="performanceChart"></canvas>
                    </div>
                </div>
                
                <!-- Asset Distribution Chart -->
                <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-200">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="text-lg font-semibold text-gray-800">Varlık Dağılımı</h3>
                        <div class="text-sm text-gray-500">Toplam: $<?php echo number_format($total_balance, 2); ?></div>
                    </div>
                    <div class="relative chart-container">
                        <canvas id="distributionChart"></canvas>
                    </div>
                </div>
            </div>
            
            <!-- Real-time Market Data -->
            <div class="bg-white rounded-xl shadow-sm p-6 border border-gray-200 mb-8">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Gerçek Zamanlı Piyasa Verileri</h3>
                    <div class="flex items-center gap-2">
                        <div class="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                        <span class="text-sm text-green-600">Canlı</span>
                    </div>
                </div>
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4" id="marketData">
                    <!-- Market data will be loaded here -->
                </div>
            </div>
            
            <!-- Holdings -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden hidden md:block">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Varlıklar</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full" id="holdingsTable">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Bakiye</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Değer (USDT)</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">24s Değişim</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php if ($active_wallets == 0): ?>
                                <tr>
                                    <td colspan="5" class="px-6 py-4 text-center text-gray-500">
                                        <div class="flex flex-col items-center gap-3">
                                            <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center">
                                                <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"></path>
                                                </svg>
                                            </div>
                                            <div>
                                                <p class="text-lg font-medium text-gray-600">Henüz varlık bulunmamaktadır</p>
                                                <p class="text-sm text-gray-500 mt-1">İlk işleminizi yapmak için <a href="trade.php" class="text-blue-600 hover:text-blue-700 font-medium">Spot İşlem</a> sayfasını ziyaret edin</p>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($wallets as $wallet): 
                                    $coin_symbol = $wallet['coin_symbol'];
                                    $balance = $wallet['balance'];
                                    
                                    // Sadece bakiyesi 0'dan büyük olanları göster
                                    if ($balance <= 0) {
                                        continue;
                                    }
                                    
                                    $price = 1;
                                    $change = 0;
                                    
                                    if (isset($crypto_prices[$coin_symbol])) {
                                        $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
                                        $change = floatval($crypto_prices[$coin_symbol]['raw_change']);
                                    }
                                    
                                    $value = $balance * $price;
                                    $changeColor = $change >= 0 ? 'text-green-600' : 'text-red-600';
                                    $sign = $change >= 0 ? '+' : '';
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <div class="flex items-center gap-3">
                                            <div class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden">
                                                <?php 
                                                    $logo = '';
                                                    if (isset($crypto_prices[$coin_symbol]) && !empty($crypto_prices[$coin_symbol]['image'])) {
                                                        $logo = $crypto_prices[$coin_symbol]['image'];
                                                    } else {
                                                        // Yerel fallback
                                                        $logo = 'images/coin-logos/' . strtolower($coin_symbol) . '.svg';
                                                    }
                                                ?>
                                                <img src="<?php echo htmlspecialchars($logo); ?>" 
                                                     alt="<?php echo htmlspecialchars($coin_symbol); ?>" 
                                                     class="w-full h-full object-cover"
                                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                                <div class="w-full h-full bg-gray-100 flex items-center justify-center font-semibold" style="display: none;">
                                                    <?php echo $coin_symbol[0]; ?>
                                                </div>
                                            </div>
                                            <div>
                                                <div class="font-semibold text-gray-800"><?php echo htmlspecialchars($coin_symbol); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-right text-gray-800 font-mono"><?php echo number_format($balance, 8); ?></td>
                                    <td class="px-6 py-4 text-right text-gray-800 font-semibold">$<?php echo number_format($value, 2); ?></td>
                                    <td class="px-6 py-4 text-right">
                                        <span class="<?php echo $changeColor; ?> font-semibold"><?php echo $sign . number_format($change, 2); ?>%</span>
                                    </td>
                                    <td class="px-6 py-4 text-right">
                                        <a href="trade.php?symbol=<?php echo $coin_symbol; ?>" class="text-blue-600 hover:text-blue-700 font-medium">İşlem Yap</a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Holdings (Mobile Cards) -->
            <div class="md:hidden">
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                    <div class="p-4 border-b border-gray-200">
                        <h2 class="text-lg font-bold text-gray-800">Varlıklar</h2>
                    </div>
                    <div class="divide-y divide-gray-100">
                        <?php if ($active_wallets == 0): ?>
                            <div class="p-6 text-center text-gray-500">Henüz varlık bulunmamaktadır.</div>
                        <?php else: ?>
                            <?php foreach ($wallets as $wallet): 
                                $coin_symbol = $wallet['coin_symbol'];
                                $balance = $wallet['balance'];
                                if ($balance <= 0) { continue; }
                                $price = isset($crypto_prices[$coin_symbol]) ? floatval($crypto_prices[$coin_symbol]['raw_price']) : 1;
                                $change = isset($crypto_prices[$coin_symbol]) ? floatval($crypto_prices[$coin_symbol]['raw_change']) : 0;
                                $value = $balance * $price;
                                $changeColor = $change >= 0 ? 'text-green-600' : 'text-red-600';
                                $sign = $change >= 0 ? '+' : '';
                                $logo = (isset($crypto_prices[$coin_symbol]) && !empty($crypto_prices[$coin_symbol]['image'])) ? $crypto_prices[$coin_symbol]['image'] : ('images/coin-logos/' . strtolower($coin_symbol) . '.svg');
                            ?>
                            <div class="p-4">
                                <div class="flex items-center gap-3">
                                    <div class="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden">
                                        <img src="<?php echo htmlspecialchars($logo); ?>" alt="<?php echo htmlspecialchars($coin_symbol); ?>" class="w-8 h-8 object-contain" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                        <div class="w-full h-full bg-gray-100 hidden items-center justify-center font-semibold"><?php echo $coin_symbol[0]; ?></div>
                                    </div>
                                    <div class="min-w-0 flex-1">
                                        <div class="flex items-center justify-between">
                                            <div class="font-semibold text-gray-800"><?php echo htmlspecialchars($coin_symbol); ?></div>
                                            <div class="text-right text-gray-800 font-semibold number-animation">$<?php echo number_format($value, 2); ?></div>
                                        </div>
                                        <div class="mt-1 grid grid-cols-3 gap-2 text-xs text-gray-600">
                                            <div><span class="text-gray-500">Bakiye:</span> <?php echo number_format($balance, 8); ?></div>
                                            <div><span class="text-gray-500">Değer:</span> $<?php echo number_format($value, 2); ?></div>
                                            <div class="<?php echo $changeColor; ?> font-medium text-right"><?php echo $sign . number_format($change, 2); ?>%</div>
                                        </div>
                                        <div class="mt-3 flex gap-2">
                                            <a href="trade.php?symbol=<?php echo $coin_symbol; ?>" class="px-3 py-1.5 rounded-md border text-xs">İşlem</a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Recent Transactions -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 mt-8 overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Son İşlemler</h2>
                </div>
                <div class="p-6" id="transactionsList">
                    <?php if (empty($recent_transactions)): ?>
                        <p class="text-gray-500 text-center">Henüz işlem bulunmamaktadır.</p>
                    <?php else: ?>
                        <div class="space-y-3">
                            <?php foreach ($recent_transactions as $tx): 
                                // Veri tipini belirle
                                $transaction_type = $tx['source'] === 'conversion' ? 'convert' : ($tx['type'] ?? 'unknown');
                                $amount = $tx['amount'] ?? 0;
                                
                                $type_map = [
                                    'deposit' => ['name' => 'Yatırma', 'color' => 'green', 'icon' => '↓', 'bg' => 'bg-green-50'],
                                    'withdraw' => ['name' => 'Çekme', 'color' => 'red', 'icon' => '↑', 'bg' => 'bg-red-50'],
                                    'transfer' => ['name' => 'Transfer', 'color' => 'blue', 'icon' => '↔', 'bg' => 'bg-blue-50'],
                                    'convert' => ['name' => 'Dönüştürme', 'color' => 'purple', 'icon' => '🔄', 'bg' => 'bg-purple-50'],
                                    'trade' => [
                                        'name' => floatval($amount) > 0 ? 'Alış İşlemi' : 'Satış İşlemi', 
                                        'color' => floatval($amount) > 0 ? 'green' : 'red',
                                        'icon' => floatval($amount) > 0 ? '↗' : '↘',
                                        'bg' => floatval($amount) > 0 ? 'bg-green-50' : 'bg-red-50'
                                    ],
                                    'staking' => ['name' => 'Stake', 'color' => 'yellow', 'icon' => '⏳', 'bg' => 'bg-yellow-50'],
                                    'reward' => ['name' => 'Ödül', 'color' => 'green', 'icon' => '🎁', 'bg' => 'bg-green-50']
                                ];
                                $type_info = $type_map[$transaction_type] ?? ['name' => ucfirst($transaction_type), 'color' => 'gray', 'icon' => '●', 'bg' => 'bg-gray-50'];
                                $status = $tx['status'] ?? 'completed';
                                $status_color = [
                                    'pending' => 'yellow',
                                    'completed' => 'green',
                                    'failed' => 'red',
                                    'cancelled' => 'gray'
                                ];
                            ?>
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                                <div class="flex items-center gap-3">
                                    <div class="w-10 h-10 rounded-full <?php echo $type_info['bg']; ?> flex items-center justify-center">
                                        <span class="text-lg"><?php echo $type_info['icon']; ?></span>
                                    </div>
                                    <div>
                                        <?php if ($tx['source'] === 'conversion'): ?>
                                            <div class="font-semibold text-gray-800"><?php echo htmlspecialchars($type_info['name']); ?> - <?php echo htmlspecialchars($tx['from_coin'] ?? ''); ?> → <?php echo htmlspecialchars($tx['to_coin'] ?? ''); ?></div>
                                            <div class="text-sm text-gray-500"><?php echo date('d.m.Y H:i', strtotime($tx['created_at'] ?? 'now')); ?></div>
                                        <?php else: ?>
                                            <div class="font-semibold text-gray-800"><?php echo htmlspecialchars($type_info['name']); ?> - <?php echo htmlspecialchars($tx['coin_symbol'] ?? ''); ?></div>
                                            <div class="text-sm text-gray-500"><?php echo date('d.m.Y H:i', strtotime($tx['created_at'] ?? 'now')); ?></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="text-right">
                                    <?php if ($tx['source'] === 'conversion'): ?>
                                        <?php 
                                        $from_coin = $tx['from_coin'] ?? '';
                                        $to_coin = $tx['to_coin'] ?? '';
                                        $from_decimals = ($from_coin === 'TRY' || $from_coin === 'USDT') ? 2 : 8;
                                        $to_decimals = ($to_coin === 'TRY' || $to_coin === 'USDT') ? 2 : 8;
                                        ?>
                                        <div class="font-semibold text-<?php echo $type_info['color']; ?>-600">
                                            <?php echo number_format($tx['from_amount'] ?? 0, $from_decimals); ?> <?php echo htmlspecialchars($from_coin); ?> → 
                                            <?php echo number_format($tx['to_amount'] ?? 0, $to_decimals); ?> <?php echo htmlspecialchars($to_coin); ?>
                                        </div>
                                        <div class="text-sm text-gray-500 number-animation">Oran: <?php echo number_format($tx['rate'] ?? 0, 6); ?></div>
                                    <?php else: ?>
                                        <?php 
                                        $coin_symbol = $tx['coin_symbol'] ?? '';
                                        $decimals = ($coin_symbol === 'TRY' || $coin_symbol === 'USDT') ? 2 : 8;
                                        ?>
                                        <div class="font-semibold text-<?php echo $type_info['color']; ?>-600"><?php echo number_format($amount, $decimals); ?> <?php echo htmlspecialchars($coin_symbol); ?></div>
                                        <div class="text-sm text-gray-500 number-animation">@ $<?php echo number_format($tx['price'] ?? 0, 2); ?></div>
                                    <?php endif; ?>
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-<?php echo $status_color[$status] ?? 'gray'; ?>-100 text-<?php echo $status_color[$status] ?? 'gray'; ?>-800">
                                        <?php echo ucfirst($status); ?>
                                    </span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>
    
    <script>
        // PHP'den gelen gerçek veriler
        const walletsData = <?php echo json_encode($wallets); ?>;
        const cryptoPrices = <?php echo json_encode($crypto_prices); ?>;
        const recentTransactions = <?php echo json_encode($recent_transactions); ?>;
        
        // Sayfa yenilendiğinde veriler otomatik PHP'den yüklenir
        console.log('Portfolio loaded successfully');
        
        // Load holdings data
        async function loadHoldings() {
            try {
                const response = await fetch('assets/crypto_prices.json');
                const data = await response.json();
                
                const tbody = document.querySelector('#holdingsTable tbody');
                tbody.innerHTML = '';
                
                // Gerçek cüzdan verilerini kullan (TRY hariç)
                const activeWallets = walletsData.filter(wallet => 
                    parseFloat(wallet.balance) > 0 && wallet.coin_symbol !== 'TRY'
                );
                
                if (activeWallets.length === 0) {
                    tbody.innerHTML = `
                        <tr>
                            <td colspan="5" class="px-6 py-8 text-center text-gray-500">
                                <div class="flex flex-col items-center gap-2">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="text-gray-400">
                                        <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3 4-3 9-3 9 1.34 9 3z"></path>
                                        <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"></path>
                                    </svg>
                                    <span class="text-lg font-medium">Henüz aktif varlığınız yok</span>
                                    <span class="text-sm">İlk işleminizi yapmak için <a href="trade.php" class="text-blue-600 hover:text-blue-700">Trade</a> sayfasını ziyaret edin</span>
                                </div>
                            </td>
                        </tr>
                    `;
                    return;
                }
                
                const holdings = activeWallets.map(wallet => ({
                    symbol: wallet.coin_symbol,
                    balance: parseFloat(wallet.balance)
                }));
                
                holdings.forEach(holding => {
                    const coinData = data.prices[holding.symbol];
                    if (coinData) {
                        const value = parseFloat(coinData.raw_price) * holding.balance;
                        const changeColor = coinData.raw_change >= 0 ? 'text-green-600' : 'text-red-600';
                        const sign = coinData.raw_change >= 0 ? '+' : '';
                        
                        // JSON'dan logo kullan, yoksa yerel fallback, en son dış kaynak
                        const logoUrl = (coinData.image && coinData.image.trim() !== '') 
                            ? coinData.image 
                            : (`images/coin-logos/${holding.symbol.toLowerCase()}.svg`);
                        
                        const tr = document.createElement('tr');
                        tr.className = 'hover:bg-gray-50';
                        tr.innerHTML = `
                            <td class="px-6 py-4">
                                <div class="flex items-center gap-3">
                                    <div class="w-10 h-10 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                        <img src="${logoUrl}" alt="${holding.symbol}" class="w-8 h-8 object-contain" 
                                             onerror="this.src='https://cryptologos.cc/logos/bitcoin-btc-logo.png'">
                                    </div>
                                    <div>
                                        <div class="font-semibold text-gray-800">${holding.symbol}</div>
                                        <div class="text-xs text-gray-500">${holding.symbol}/USDT</div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4 text-right text-gray-800 font-mono">${holding.balance}</td>
                            <td class="px-6 py-4 text-right text-gray-800 font-semibold number-animation">$${value.toFixed(2)}</td>
                            <td class="px-6 py-4 text-right ${changeColor} font-semibold">${sign}${coinData.change}%</td>
                            <td class="px-6 py-4 text-right">
                                <a href="trade.php" target="_blank" class="text-blue-600 hover:text-blue-700 text-sm font-medium">Trade</a>
                            </td>
                        `;
                        tbody.appendChild(tr);
                    }
                });
                // Toplam bakiyeyi güncelle
                updateTotalBalance();
                
            } catch (error) {
                console.error('Holdings yüklenemedi:', error);
            }
        }
        
        // Load transactions
        function loadTransactions() {
            const transactionsList = document.getElementById('transactionsList');
            
            if (!transactionsList) {
                console.error('Transactions list element bulunamadı');
                return;
            }
            
            if (!recentTransactions || recentTransactions.length === 0) {
                transactionsList.innerHTML = `
                    <div class="text-center py-8 text-gray-500">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="mx-auto mb-4 text-gray-400">
                            <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3 4-3 9-3 9 1.34 9 3z"></path>
                            <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"></path>
                        </svg>
                        <span class="text-lg font-medium">Henüz işlem geçmişiniz yok</span>
                    </div>
                `;
                return;
            }
            
            // Dönüştürme işlemlerini filtrele (sadece pozitif olanları göster)
            const filteredTransactions = recentTransactions.filter(tx => {
                if (tx.source === 'conversion') {
                    return true; // Dönüştürme işlemlerini göster
                }
                
                // Normal işlemler için, eğer aynı tarihte dönüştürme varsa gösterme
                const txDate = new Date(tx.created_at).toDateString();
                const hasConversion = recentTransactions.some(conv => 
                    conv.source === 'conversion' && 
                    new Date(conv.created_at).toDateString() === txDate
                );
                
                return !hasConversion; // Dönüştürme varsa normal işlemi gösterme
            });
            
            transactionsList.innerHTML = filteredTransactions.map(tx => {
                // İşlem türünü belirle
                let transactionType = tx.type;
                if (tx.source === 'conversion') {
                    transactionType = 'convert';
                } else if (!transactionType || transactionType === '') {
                    // Type boşsa, amount'a göre tahmin et
                    const amount = parseFloat(tx.amount || 0);
                    if (amount > 0) {
                        transactionType = 'deposit'; // Pozitif miktar = yatırma
                    } else if (amount < 0) {
                        transactionType = 'withdraw'; // Negatif miktar = çekme
                    } else {
                        transactionType = 'unknown';
                    }
                }
                
                // İşlem türü mapping'i
                const typeMap = {
                    'deposit': { name: 'Yatırma', color: 'text-green-600', icon: '↓', bgColor: 'bg-green-50' },
                    'withdraw': { name: 'Çekme', color: 'text-red-600', icon: '↑', bgColor: 'bg-red-50' },
                    'transfer': { name: 'Transfer', color: 'text-blue-600', icon: '↔', bgColor: 'bg-blue-50' },
                    'convert': { name: 'Dönüştürme', color: 'text-purple-600', icon: '🔄', bgColor: 'bg-purple-50' },
                    'trade': { 
                        name: parseFloat(tx.amount || 0) > 0 ? 'Alış İşlemi' : 'Satış İşlemi', 
                        color: parseFloat(tx.amount || 0) > 0 ? 'text-green-600' : 'text-red-600',
                        icon: parseFloat(tx.amount || 0) > 0 ? '↗' : '↘',
                        bgColor: parseFloat(tx.amount || 0) > 0 ? 'bg-green-50' : 'bg-red-50'
                    },
                    'staking': { name: 'Stake', color: 'text-yellow-600', icon: '⏳', bgColor: 'bg-yellow-50' },
                    'reward': { name: 'Ödül', color: 'text-green-600', icon: '🎁', bgColor: 'bg-green-50' },
                    'unknown': { name: 'Bilinmeyen', color: 'text-gray-600', icon: '?', bgColor: 'bg-gray-50' }
                };
                
                const typeInfo = typeMap[transactionType] || { name: (transactionType || 'Unknown').toUpperCase(), color: 'text-gray-600', icon: '●', bgColor: 'bg-gray-50' };
                const date = new Date(tx.created_at || new Date()).toLocaleString('tr-TR');
                const coinSymbol = tx.coin_symbol || 'UNKNOWN';
                const amount = parseFloat(tx.amount || 0);
                const price = parseFloat(tx.price || 0);
                const logo = (cryptoPrices[coinSymbol] && cryptoPrices[coinSymbol].image) 
                    ? cryptoPrices[coinSymbol].image 
                    : (`images/coin-logos/${coinSymbol.toLowerCase()}.svg`);
                
                return `
                    <div class="flex items-center justify-between py-3 border-b border-gray-100 last:border-0">
                        <div class="flex items-center gap-4">
                            <div class="w-10 h-10 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                <img src="${logo}" alt="${coinSymbol}" class="w-8 h-8 object-contain" 
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="w-full h-full bg-gray-100 hidden items-center justify-center font-semibold">${coinSymbol[0]}</div>
                            </div>
                            <div>
                                ${tx.source === 'conversion' 
                                    ? `<div class="font-semibold text-gray-800">${typeInfo.name} - ${tx.from_coin || 'UNKNOWN'} → ${tx.to_coin || 'UNKNOWN'}</div>`
                                    : `<div class="font-semibold text-gray-800">${typeInfo.name} - ${coinSymbol}</div>`
                                }
                                <div class="text-xs text-gray-500">${date}</div>
                            </div>
                        </div>
                        <div class="text-right">
                            ${tx.source === 'conversion' 
                                ? `<div class="font-semibold ${typeInfo.color}">${formatBalance(tx.from_amount || 0, tx.from_coin || 'UNKNOWN')} ${tx.from_coin || 'UNKNOWN'} → ${formatBalance(tx.to_amount || 0, tx.to_coin || 'UNKNOWN')} ${tx.to_coin || 'UNKNOWN'}</div>
                                   <div class="text-xs text-gray-500 number-animation">Oran: ${(parseFloat(tx.rate) || 0).toFixed(6)}</div>`
                                : `<div class="font-semibold ${typeInfo.color}">${formatBalance(amount, coinSymbol)} ${coinSymbol}</div>
                                   <div class="text-xs text-gray-500 number-animation">@ $${price.toFixed(2)}</div>`
                            }
                        </div>
                    </div>
                `;
            }).join('');
        }
        
        // Toplam bakiyeyi güncelle
        function formatBalance(amount, symbol) {
            // Amount'u sayıya çevir
            amount = parseFloat(amount) || 0;
            
            if (amount === 0) return '0';
            
            // TRY için özel format
            if (symbol === 'TRY') {
                return amount.toFixed(2);
            }
            
            // USDT için özel format (kısaltma yok)
            if (symbol === 'USDT') {
                return amount.toFixed(2);
            }
            
            // Diğer coinler için akıllı format
            if (amount >= 1000000) {
                return (amount / 1000000).toFixed(2) + 'M';
            } else if (amount >= 1000) {
                return (amount / 1000).toFixed(2) + 'K';
            } else if (amount >= 1) {
                return amount.toFixed(4);
            } else if (amount >= 0.01) {
                return amount.toFixed(6);
            } else {
                return amount.toFixed(8);
            }
        }

        function updateTotalBalance() {
            let totalBalance = 0;
            let availableBalance = 0;
            
            walletsData.forEach(wallet => {
                const balance = parseFloat(wallet.balance);
                const available = parseFloat(wallet.available);
                
                // TRY'yi toplam bakiyeden hariç tut
                if (wallet.coin_symbol === 'TRY') {
                    return; // TRY'yi atla
                }
                
                if (balance > 0) {
                    // Fiyatı cryptoPrices'dan al
                    const price = cryptoPrices[wallet.coin_symbol] ? 
                        parseFloat(cryptoPrices[wallet.coin_symbol].raw_price) : 1;
                    
                    totalBalance += balance * price;
                    availableBalance += available * price;
                }
            });
            
            // UI'da göster
            const totalElement = document.querySelector('.text-3xl.font-bold.text-gray-800');
            const availableElement = document.querySelector('.text-3xl.font-bold.text-gray-800:nth-of-type(2)');
            
            if (totalElement) {
                totalElement.textContent = '$' + totalBalance.toFixed(2);
                totalElement.classList.add('number-animation');
            }
            if (availableElement) {
                availableElement.textContent = '$' + availableBalance.toFixed(2);
                availableElement.classList.add('number-animation');
            }
        }
        
        // Gerçek zamanlı güncelleme başlat
        function startRealTimeUpdates() {
            // Her 30 saniyede bir portföy verilerini güncelle
            setInterval(async () => {
                try {
                    const response = await fetch('get_portfolio_data.php');
                    const data = await response.json();
                    
                    if (data.success) {
                        // Fiyatları güncelle
                        Object.keys(data.data.crypto_prices).forEach(symbol => {
                            cryptoPrices[symbol] = data.data.crypto_prices[symbol];
                        });
                        
                        // Holdings ve bakiyeleri yeniden yükle
                        await loadHoldings();
                        
                        // Grafikleri güncelle
                        updateCharts();
                        
                        console.log('Portföy verileri güncellendi');
                    }
                    
                } catch (error) {
                    console.error('Gerçek zamanlı güncelleme hatası:', error);
                }
            }, 30000); // 30 saniye
            
            // Her 5 saniyede bir piyasa verilerini güncelle
            setInterval(updateMarketData, 5000);
            
            // Her 10 saniyede bir grafikleri güncelle
            setInterval(updateCharts, 10000);
        }
        
        // Grafikleri başlat
        let performanceChart, distributionChart;
        
        async function initializeCharts() {
            try {
                console.log('Grafikler başlatılıyor...');
                
                // Performans Grafiği
                const performanceCtx = document.getElementById('performanceChart');
                if (!performanceCtx) {
                    console.error('Performance chart canvas bulunamadı');
                    return;
                }
                
                console.log('Portfolio verisi alınıyor...');
                const portfolioData = await generatePortfolioData('7d');
                console.log('Portfolio verisi:', portfolioData);
            
            performanceChart = new Chart(performanceCtx.getContext('2d'), {
                type: 'line',
                data: {
                    labels: generateTimeLabels('7d'),
                    datasets: [{
                        label: 'Portföy Değeri',
                        data: portfolioData,
                        borderColor: '#3B82F6',
                        backgroundColor: 'rgba(59, 130, 246, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: false,
                            grid: {
                                color: '#F3F4F6'
                            }
                        },
                        x: {
                            grid: {
                                color: '#F3F4F6'
                            }
                        }
                    },
                    layout: {
                        padding: {
                            top: 10,
                            bottom: 10,
                            left: 10,
                            right: 10
                        }
                    }
                }
            });
            
            // Dağılım Grafiği
            console.log('Dağılım grafiği oluşturuluyor...');
            const distributionCtx = document.getElementById('distributionChart');
            if (!distributionCtx) {
                console.error('Distribution chart canvas bulunamadı');
                return;
            }
            
            const assetLabels = generateAssetLabels();
            const assetData = generateAssetData();
            console.log('Asset labels:', assetLabels);
            console.log('Asset data:', assetData);
            
            distributionChart = new Chart(distributionCtx.getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: generateAssetLabels(),
                    datasets: [{
                        data: generateAssetData(),
                        backgroundColor: [
                            '#3B82F6', '#EF4444', '#10B981', '#F59E0B', '#8B5CF6',
                            '#06B6D4', '#84CC16', '#F97316', '#EC4899', '#6366F1'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 20,
                                usePointStyle: true
                            }
                        }
                    },
                    layout: {
                        padding: {
                            top: 10,
                            bottom: 10,
                            left: 10,
                            right: 10
                        }
                    }
                }
            });
            
            } catch (error) {
                console.error('Grafikler yüklenirken hata oluştu:', error);
            }
        }
        
        // Grafik verilerini oluştur
        function generateTimeLabels(period) {
            const labels = [];
            const now = new Date();
            const days = period === '7d' ? 7 : period === '30d' ? 30 : 90;
            
            for (let i = days; i >= 0; i--) {
                const date = new Date(now);
                date.setDate(date.getDate() - i);
                labels.push(date.toLocaleDateString('tr-TR', { month: 'short', day: 'numeric' }));
            }
            return labels;
        }
        
        async function generatePortfolioData(period) {
            try {
                const response = await fetch(`get_portfolio_data.php?period=${period}`);
                const data = await response.json();
                
                if (data.success) {
                    return data.data.portfolio_performance.map(item => item.value);
                }
            } catch (error) {
                console.error('Portföy performans verisi alınamadı:', error);
            }
            
            // Fallback: simüle edilmiş veri
            const data = [];
            const baseValue = <?php echo $total_balance; ?>;
            const days = period === '7d' ? 7 : period === '30d' ? 30 : 90;
            
            for (let i = 0; i <= days; i++) {
                const variation = (Math.random() - 0.5) * 0.1; // ±5% varyasyon
                const value = baseValue * (1 + variation * (i / days));
                data.push(value.toFixed(2));
            }
            return data;
        }
        
        function generateAssetLabels() {
            return walletsData
                .filter(wallet => parseFloat(wallet.balance) > 0)
                .map(wallet => wallet.coin_symbol);
        }
        
        function generateAssetData() {
            return walletsData
                .filter(wallet => parseFloat(wallet.balance) > 0)
                .map(wallet => {
                    let price = 1;
                    
                    if (wallet.coin_symbol === 'TRY') {
                        // TRY için özel fiyat (1 TRY = 0.033 USD)
                        price = 0.033;
                    } else if (cryptoPrices[wallet.coin_symbol]) {
                        price = parseFloat(cryptoPrices[wallet.coin_symbol].raw_price);
                    }
                    
                    return (parseFloat(wallet.balance) * price).toFixed(2);
                });
        }
        
        // Grafik güncelleme fonksiyonları
        async function updateChart(period) {
            if (performanceChart) {
                const portfolioData = await generatePortfolioData(period);
                performanceChart.data.labels = generateTimeLabels(period);
                performanceChart.data.datasets[0].data = portfolioData;
                performanceChart.update();
            }
            
            // Buton durumlarını güncelle
            document.querySelectorAll('[onclick^="updateChart"]').forEach(btn => {
                btn.className = 'px-3 py-1 text-xs bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200';
            });
            event.target.className = 'px-3 py-1 text-xs bg-blue-100 text-blue-600 rounded-full hover:bg-blue-200';
        }
        
        function updateCharts() {
            if (distributionChart) {
                distributionChart.data.labels = generateAssetLabels();
                distributionChart.data.datasets[0].data = generateAssetData();
                distributionChart.update();
            }
        }
        
        // Gerçek zamanlı piyasa verileri
        async function updateMarketData() {
            try {
                const response = await fetch('get_portfolio_data.php');
                const data = await response.json();
                
                if (data.success) {
                    const marketDataContainer = document.getElementById('marketData');
                    const marketData = data.data.market_data;
                    
                    marketDataContainer.innerHTML = marketData.map(coin => {
                        const change = parseFloat(coin.change);
                        const isPositive = change >= 0;
                        const changeColor = isPositive ? 'text-green-600' : 'text-red-600';
                        const changeIcon = isPositive ? '↗' : '↘';
                        
                        return `
                            <div class="bg-gray-50 rounded-lg p-4 hover:bg-gray-100 transition-colors">
                                <div class="flex items-center justify-between mb-2">
                                    <span class="font-semibold text-gray-800">${coin.symbol}</span>
                                    <span class="text-xs ${changeColor}">${changeIcon} ${Math.abs(change).toFixed(2)}%</span>
                                </div>
                                <div class="text-lg font-bold text-gray-900">$${parseFloat(coin.price).toLocaleString()}</div>
                                <div class="text-xs text-gray-500">24s Vol: ${coin.volume}</div>
                            </div>
                        `;
                    }).join('');
                }
            } catch (error) {
                console.error('Piyasa verileri güncelleme hatası:', error);
            }
        }
        
        // Sayfa yüklendiğinde başlat
        document.addEventListener('DOMContentLoaded', function() {
            loadHoldings();
            loadTransactions();
            updateTotalBalance();
            startRealTimeUpdates();
            initializeCharts();
            updateMarketData();
        });
        
    </script>
</body>
</html>

