<?php 
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];

// Kullanıcının stake'lerini al
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM stakes WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$user_id]);
    $user_stakes = $stmt->fetchAll();
} catch (Exception $e) {
    $user_stakes = [];
}

// Kullanıcının cüzdanlarını al
try {
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ? AND balance > 0 ORDER BY balance DESC");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
} catch (Exception $e) {
    $wallets = [];
}

include 'components/header.php';
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stake - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-8">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">Stake</h1>
                <p class="text-gray-600">Kripto para birimlerinizi stake ederek ödül kazanın</p>
            </div>

            <!-- Staking Overview -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Stake</p>
                            <p class="text-2xl font-bold text-gray-900" id="totalStaked">$0.00</p>
                        </div>
                        <div class="p-3 bg-blue-100 rounded-full">
                            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Ödüller</p>
                            <p class="text-2xl font-bold text-green-600" id="totalRewards">$0.00</p>
                        </div>
                        <div class="p-3 bg-green-100 rounded-full">
                            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Aktif Stake'ler</p>
                            <p class="text-2xl font-bold text-purple-600" id="activeStakes">0</p>
                        </div>
                        <div class="p-3 bg-purple-100 rounded-full">
                            <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">APY</p>
                            <p class="text-2xl font-bold text-orange-600" id="averageAPY">0.00%</p>
                        </div>
                        <div class="p-3 bg-orange-100 rounded-full">
                            <svg class="w-6 h-6 text-orange-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Available Staking Pools -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden mb-8">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Mevcut Stake Havuzları</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">APY</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Min. Miktar</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Kilit Süresi</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">İşlem</th>
                            </tr>
                        </thead>
                        <tbody id="stakingPoolsTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- Staking pools will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- My Stakes -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Stake'lerim</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Stake Miktarı</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">APY</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Ödüller</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Durum</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">İşlem</th>
                            </tr>
                        </thead>
                        <tbody id="myStakesTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- My stakes will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Staking Modal -->
    <div id="stakingModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800"><span id="stakeAssetName">BTC</span> Stake Et</h3>
                    <button onclick="hideStakingModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Stake Edilecek Miktar</label>
                        <input type="number" id="stakeAmount" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Kilit Süresi</label>
                        <select id="lockPeriod" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="30">30 Gün</option>
                            <option value="90">90 Gün</option>
                            <option value="180">180 Gün</option>
                            <option value="365">365 Gün</option>
                        </select>
                    </div>
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-3">
                        <p class="text-sm text-blue-800">Beklenen APY: <span id="expectedAPY">5.2%</span></p>
                        <p class="text-sm text-blue-800 text-gray-500">Günlük gelir hesaplaması için kısa süre bekleyin.</p>
                    </div>
                    <button onclick="confirmStake()" class="w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700">Şimdi Stake Et</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let stakingPools = [];
        let myStakes = [];
        let cryptoPrices = {};
        
        // PHP'den gelen stake verilerini JavaScript'e aktar
        const userStakesFromPHP = <?php echo json_encode($user_stakes); ?>;
        
        async function loadStakingData() {
            try {
                // Load crypto prices from admin-managed JSON
                const response = await fetch('assets/crypto_prices.json?t=' + Date.now());
                const data = await response.json();
                
                if (data && data.prices) {
                    cryptoPrices = data.prices;
                    
                    // Create staking pools from available coins
                    stakingPools = Object.entries(data.prices).map(([symbol, priceData]) => ({
                        asset: symbol,
                        name: priceData.name || symbol,
                        apy: Math.random() * 15 + 2, // Random APY between 2-17%
                        minAmount: symbol === 'BTC' ? 0.01 : symbol === 'ETH' ? 0.1 : 1,
                        lockPeriod: '30-365 days',
                        logo: priceData.image || null
                    }));
                } else {
                    // Fallback to demo data
                    stakingPools = [
                        { asset: 'BTC', name: 'Bitcoin', apy: 5.2, minAmount: 0.01, lockPeriod: '30-365 days', logo: null },
                        { asset: 'ETH', name: 'Ethereum', apy: 4.8, minAmount: 0.1, lockPeriod: '30-365 days', logo: null },
                        { asset: 'SOL', name: 'Solana', apy: 6.5, minAmount: 1, lockPeriod: '30-365 days', logo: null },
                        { asset: 'ADA', name: 'Cardano', apy: 4.2, minAmount: 10, lockPeriod: '30-365 days', logo: null },
                        { asset: 'DOT', name: 'Polkadot', apy: 12.5, minAmount: 1, lockPeriod: '30-365 days', logo: null }
                    ];
                }
            } catch (error) {
                console.error('Staking data yüklenemedi:', error);
                // Fallback to demo data
                stakingPools = [
                    { asset: 'BTC', name: 'Bitcoin', apy: 5.2, minAmount: 0.01, lockPeriod: '30-365 days', logo: null },
                    { asset: 'ETH', name: 'Ethereum', apy: 4.8, minAmount: 0.1, lockPeriod: '30-365 days', logo: null },
                    { asset: 'SOL', name: 'Solana', apy: 6.5, minAmount: 1, lockPeriod: '30-365 days', logo: null },
                    { asset: 'ADA', name: 'Cardano', apy: 4.2, minAmount: 10, lockPeriod: '30-365 days', logo: null },
                    { asset: 'DOT', name: 'Polkadot', apy: 12.5, minAmount: 1, lockPeriod: '30-365 days', logo: null }
                ];
            }
            
            // PHP'den gelen gerçek stake verilerini kullan
            myStakes = userStakesFromPHP.map(stake => ({
                id: stake.id,
                asset: stake.coin_symbol,
                name: stake.coin_symbol,
                stakedAmount: parseFloat(stake.amount),
                apy: parseFloat(stake.apy),
                rewards: parseFloat(stake.rewards),
                status: stake.status,
                lockPeriod: stake.lock_period,
                unlockDate: stake.unlock_date,
                createdDate: stake.created_at
            }));
            
            updateStakingPoolsTable();
            updateMyStakesTable();
            updateOverview();
        }

        function updateStakingPoolsTable() {
            const tbody = document.getElementById('stakingPoolsTableBody');
            tbody.innerHTML = '';

            stakingPools.forEach(pool => {
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                ${pool.logo ? `<img src="${pool.logo}" alt="${pool.asset}" class="w-6 h-6 object-contain">` : `<div class="w-6 h-6 rounded-full bg-gray-200 text-gray-700 text-[10px] font-bold flex items-center justify-center">${pool.asset.substring(0,2)}</div>`}
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${pool.asset}</div>
                                <div class="text-xs text-gray-500">${pool.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${parseFloat(pool.apy).toFixed(1)}%</td>
                    <td class="px-6 py-4 text-right text-gray-800">${pool.minAmount} ${pool.asset}</td>
                    <td class="px-6 py-4 text-right text-gray-600">${pool.lockPeriod}</td>
                    <td class="px-6 py-4 text-center">
                        <button onclick="showStakingModal('${pool.asset}', '${pool.name}', ${pool.apy})" 
                                class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm">
                            Stake
                        </button>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateMyStakesTable() {
            const tbody = document.getElementById('myStakesTableBody');
            tbody.innerHTML = '';

            if (myStakes.length === 0) {
                tbody.innerHTML = '<tr><td colspan="6" class="px-6 py-8 text-center text-gray-500">Aktif stake yok</td></tr>';
                return;
            }

            myStakes.forEach(stake => {
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full overflow-hidden bg-gray-100 flex items-center justify-center">
                                ${stake.logo ? `<img src="${stake.logo}" alt="${stake.asset}" class="w-6 h-6 object-contain">` : `<div class="w-6 h-6 rounded-full bg-gray-200 text-gray-700 text-[10px] font-bold flex items-center justify-center">${stake.asset.substring(0,2)}</div>`}
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${stake.asset}</div>
                                <div class="text-xs text-gray-500">${stake.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${stake.stakedAmount} ${stake.asset}</td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${parseFloat(stake.apy).toFixed(1)}%</td>
                    <td class="px-6 py-4 text-right text-green-600 font-semibold">${stake.rewards} ${stake.asset}</td>
                    <td class="px-6 py-4">
                        <span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">
                            Aktif
                        </span>
                    </td>
                    <td class="px-6 py-4 text-center">
                        <div class="flex justify-center gap-2">
                            <button onclick="claimRewards(${stake.id})" class="text-green-600 hover:text-green-700 text-sm">Topla</button>
                            <button onclick="unstake(${stake.id}, '${stake.asset}')" class="text-red-600 hover:text-red-700 text-sm">Çek</button>
                        </div>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateOverview() {
            let totalStaked = 0;
            let totalRewards = 0;
            let activeStakes = myStakes.filter(stake => stake.status === 'active' || stake.status === 'locked').length;
            let totalAPY = 0;

            myStakes.forEach(stake => {
                if (stake.status === 'active' || stake.status === 'locked') {
                    // Gerçek değerleri hesapla
                    const price = cryptoPrices[stake.asset] ? parseFloat(cryptoPrices[stake.asset].raw_price) : 0;
                    totalStaked += stake.stakedAmount * price;
                    totalRewards += stake.rewards * price;
                    totalAPY += stake.apy;
                }
            });

            const averageAPY = activeStakes > 0 ? totalAPY / activeStakes : 0;

            document.getElementById('totalStaked').textContent = `$${totalStaked.toFixed(2)}`;
            document.getElementById('totalRewards').textContent = `$${totalRewards.toFixed(2)}`;
            document.getElementById('activeStakes').textContent = activeStakes.toString();
            document.getElementById('averageAPY').textContent = `${averageAPY.toFixed(1)}%`;
        }

        function showStakingModal(asset, name, apy) {
            document.getElementById('stakeAssetName').textContent = asset;
            document.getElementById('expectedAPY').textContent = `${apy}%`;
            document.getElementById('stakingModal').classList.remove('hidden');
        }

        function hideStakingModal() {
            document.getElementById('stakingModal').classList.add('hidden');
        }

        function confirmStake() {
            const amount = document.getElementById('stakeAmount').value;
            const lockPeriod = document.getElementById('lockPeriod').value;
            const asset = document.getElementById('stakeAssetName').textContent;
            const apy = document.getElementById('expectedAPY').textContent.replace('%', '');

            if (!amount || amount <= 0) {
                alert('Lütfen geçerli bir miktar girin');
                return;
            }

            // FormData oluştur
            const formData = new FormData();
            formData.append('coin_symbol', asset);
            formData.append('amount', amount);
            formData.append('lock_period', lockPeriod);
            formData.append('apy', apy);

            // API'ye gönder
            fetch('api/process_stake.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(data.message);
                    hideStakingModal();
                    loadStakingData();
                } else {
                    alert('Hata: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu!');
            });
        }

        function claimRewards(stakeId) {
            const formData = new FormData();
            formData.append('stake_id', stakeId);

            fetch('api/process_claim.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(data.message);
                    loadStakingData();
                } else {
                    alert('Hata: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Bir hata oluştu!');
            });
        }

        function unstake(stakeId, asset) {
            if (confirm(`${asset} stake'ini çekmek istediğinizden emin misiniz?`)) {
                const formData = new FormData();
                formData.append('stake_id', stakeId);

                fetch('api/process_unstake.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert(data.message);
                        loadStakingData();
                    } else {
                        alert('Hata: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Bir hata oluştu!');
                });
            }
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadStakingData();
        });
    </script>

    <?php include 'components/footer.php'; ?>
</body>
</html>
