<?php
/**
 * Kriptocum - Otomatik Fiyat Güncelleme Script
 * Bu script Binance API'den güncel kripto fiyatlarını çeker ve JSON dosyasına kaydeder
 * Cron job ile her 30 saniyede bir çalıştırılabilir
 */

// Binance API endpoints
$coins = [
    'BTC' => 'BTCUSDT',
    'ETH' => 'ETHUSDT',
    'SOL' => 'SOLUSDT',
    'BNB' => 'BNBUSDT',
    'XRP' => 'XRPUSDT',
    'ADA' => 'ADAUSDT',
    'DOGE' => 'DOGEUSDT',
    'MATIC' => 'MATICUSDT',
    'AVAX' => 'AVAXUSDT',
    'LINK' => 'LINKUSDT',
    'DOT' => 'DOTUSDT',
    'UNI' => 'UNIUSDT',
    'ATOM' => 'ATOMUSDT',
    'FIL' => 'FILUSDT',
    'LTC' => 'LTCUSDT',
    'BCH' => 'BCHUSDT',
    'EOS' => 'EOSUSDT',
    'NEAR' => 'NEARUSDT',
    'APT' => 'APTUSDT',
    'OP' => 'OPUSDT',
    'ARB' => 'ARBUSDT',
    'INJ' => 'INJUSDT',
    'HBAR' => 'HBARUSDT',
    'FTM' => 'FTMUSDT',
    'RUNE' => 'RUNEUSDT',
    'ALGO' => 'ALGOUSDT',
    'GRT' => 'GRTUSDT',
    'AAVE' => 'AAVEUSDT',
    'MKR' => 'MKRUSDT',
    'COMP' => 'COMPUSDT',
    'SAND' => 'SANDUSDT',
    'MANA' => 'MANAUSDT',
    'AXS' => 'AXSUSDT',
    'CHZ' => 'CHZUSDT',
    'ENJ' => 'ENJUSDT',
    'GALA' => 'GALAUSDT',
    'ZEC' => 'ZECUSDT',
    'VET' => 'VETUSDT',
    'THETA' => 'THETAUSDT',
    'ZIL' => 'ZILUSDT',
    'ONE' => 'ONEUSDT',
    'EGLD' => 'EGLDUSDT',
    'IOTA' => 'IOTAUSDT',
    'FLOW' => 'FLOWUSDT',
    'KLAY' => 'KLAYUSDT',
];

$prices = [];
$existingPrices = [];
$dataFile = __DIR__ . '/assets/crypto_prices.json';

// Var olan dosyadan mevcut verileri (özellikle image alanlarını) oku
if (file_exists($dataFile)) {
    $existingData = json_decode(@file_get_contents($dataFile), true);
    if (is_array($existingData) && isset($existingData['prices']) && is_array($existingData['prices'])) {
        $existingPrices = $existingData['prices'];
    }
}
$updateTime = time();

// Binance API'den fiyatları çek
foreach ($coins as $coinSymbol => $tradingPair) {
    $url = "https://api.binance.com/api/v3/ticker/24hr?symbol={$tradingPair}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        $data = json_decode($response, true);
        
        if ($data && isset($data['lastPrice']) && isset($data['priceChangePercent'])) {
            $price = floatval($data['lastPrice']);
            $change = floatval($data['priceChangePercent']);
            $volume = floatval($data['volume']);
            
            // Volume değerini kısalt
            if ($volume >= 1000000000) {
                $volumeStr = '$' . round($volume / 1000000000, 2) . 'B';
            } elseif ($volume >= 1000000) {
                $volumeStr = '$' . round($volume / 1000000, 2) . 'M';
            } else {
                $volumeStr = '$' . round($volume / 1000, 2) . 'K';
            }
            
            $prices[$coinSymbol] = [
                'symbol' => $coinSymbol . '/USDT',
                'price' => number_format($price, 8, '.', ','),
                'change' => number_format($change, 2, '.', ','),
                'volume' => $volumeStr,
                'raw_price' => $price,
                'raw_change' => $change
            ];

            // Mevcut görseli koru (admin/coins.php üzerinden yüklenen)
            if (isset($existingPrices[$coinSymbol]['image']) && !empty($existingPrices[$coinSymbol]['image'])) {
                $prices[$coinSymbol]['image'] = $existingPrices[$coinSymbol]['image'];
            }
        }
    }
    
    // Rate limiting için kısa bir bekleme
    usleep(100000); // 0.1 saniye
}

// USDTRY kurunu çek ve TRY/USD oranını hesapla
function fetchUsdTryRate(): ?float {
    $url = 'https://api.exchangerate.host/latest?base=USD&symbols=TRY';
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($httpCode === 200 && $response) {
        $data = json_decode($response, true);
        if (isset($data['rates']['TRY'])) {
            return floatval($data['rates']['TRY']); // USD -> TRY
        }
    }
    return null;
}

// TRY'yi fiyat listesine ekle (raw_price = 1 TRY'nin USD karşılığı)
// USDTRY başarısız olursa son yazılan dosyadan ya da sabit fallback kullan
$usdTry = fetchUsdTryRate();
$tryUsd = null;
if ($usdTry && $usdTry > 0) {
    $tryUsd = 1.0 / $usdTry;
} else {
    // Son dosyadan dene
    $existingFile = __DIR__ . '/assets/crypto_prices.json';
    if (file_exists($existingFile)) {
        $existing = json_decode(file_get_contents($existingFile), true);
        if (isset($existing['prices']['TRY']['raw_price'])) {
            $tryUsd = floatval($existing['prices']['TRY']['raw_price']);
        }
    }
    if (!$tryUsd) {
        $tryUsd = 1.0 / 42.05; // fallback ~ 0.02378 (1 USD = 42.05 TRY)
    }
}

$prices['TRY'] = [
    'symbol' => 'TRY/USD',
    'price' => number_format($tryUsd, 8, '.', ','),
    'change' => '0.00',
    'volume' => '-',
    'raw_price' => $tryUsd,
    'raw_change' => 0
];

// TRY için mevcut görseli koru (varsa)
if (isset($existingPrices['TRY']['image']) && !empty($existingPrices['TRY']['image'])) {
    $prices['TRY']['image'] = $existingPrices['TRY']['image'];
}

// Listede olmayan (özel eklenmiş) coinleri tamamen koru
foreach ($existingPrices as $symbol => $entry) {
    if (!isset($prices[$symbol])) {
        $prices[$symbol] = $entry; // tüm alanlarıyla koru (image dahil)
    }
}

// Sonuçları JSON dosyasına kaydet
$result = [
    'last_update' => date('Y-m-d H:i:s'),
    'timestamp' => $updateTime,
    'prices' => $prices
];

$jsonData = json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
$written = file_put_contents($dataFile, $jsonData);

// Konsol çıktısı
echo "✅ Fiyatlar güncellendi: " . date('Y-m-d H:i:s') . "\n";
echo "📊 Toplam coin: " . count($prices) . "\n";
echo "💾 Kaydedilen dosya: assets/crypto_prices.json\n";
echo "🔢 Yazılan byte: " . $written . "\n";

if ($written === false) {
    echo "❌ HATA: Dosya yazılamadı!\n";
} else {
    echo "✅ Başarılı!\n";
}
?>

