<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];

// Kullanıcının işlem geçmişini veritabanından çek
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 100");
    $stmt->execute([$user_id]);
    $transactions = $stmt->fetchAll();
} catch (Exception $e) {
    $transactions = [];
}

include 'components/header.php';
?>
<?php
function format_tx_amount($symbol, $amount) {
    $fiatOrStable = ['TRY','USD','USDT','USDC','EUR'];
    if (in_array(strtoupper($symbol), $fiatOrStable, true)) {
        return number_format((float)$amount, 2, '.', ',');
    }
    $val = number_format((float)$amount, 8, '.', '');
    $val = rtrim($val, '0');
    $val = rtrim($val, '.');
    return $val === '' ? '0' : $val;
}
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cüzdan Geçmişi - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-6">
            <!-- Header -->
            <div class="mb-6 md:mb-8">
                <div class="flex items-center gap-3 md:gap-4 mb-3 md:mb-4">
                    <button onclick="window.history.back()" class="p-2 hover:bg-gray-100 rounded-lg">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                        </svg>
                    </button>
                    <div>
                        <h1 class="text-2xl md:text-3xl font-bold text-gray-800 mb-1 md:mb-2">Cüzdan Geçmişi</h1>
                        <p class="text-sm md:text-base text-gray-600">İşlem geçmişi ve cüzdan aktiviteleri</p>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6 mb-6 sticky top-14 z-10">
                <div class="grid grid-cols-1 md:grid-cols-4 gap-3 md:gap-4">
                    <!-- Transaction Type -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Tür</label>
                        <select id="typeFilter" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="all">Tüm Türler</option>
                            <option value="deposit">Yatır</option>
                            <option value="withdraw">Çek</option>
                            <option value="transfer">Aktar</option>
                            <option value="trade">İşlem</option>
                        </select>
                    </div>

                    <!-- Asset -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Varlık</label>
                        <select id="assetFilter" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="all">Tüm Varlıklar</option>
                            <option value="BTC">BTC</option>
                            <option value="ETH">ETH</option>
                            <option value="USDT">USDT</option>
                            <option value="SOL">SOL</option>
                        </select>
                    </div>

                    <!-- Date Range -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Başlangıç Tarihi</label>
                        <input type="date" id="startDate" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Bitiş Tarihi</label>
                        <input type="date" id="endDate" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                </div>

                <div class="mt-3 md:mt-4 flex flex-wrap gap-2">
                    <button onclick="applyFilters()" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm">
                        Filtreleri Uygula
                    </button>
                    <button onclick="resetFilters()" class="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors text-sm">
                        Sıfırla
                    </button>
                    <button onclick="exportHistory()" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm">
                        CSV Dışa Aktar
                    </button>
                </div>
            </div>

            <!-- Transaction Summary -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 md:gap-6 mb-6">
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Yatırımlar</p>
                            <p class="text-2xl font-bold text-green-600" id="totalDeposits">$0.00</p>
                        </div>
                        <div class="p-3 bg-green-100 rounded-full">
                            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Çekimler</p>
                            <p class="text-2xl font-bold text-red-600" id="totalWithdrawals">$0.00</p>
                        </div>
                        <div class="p-3 bg-red-100 rounded-full">
                            <svg class="w-6 h-6 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4m16 0l-4-4m4 4l-4 4"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Total Transfers</p>
                            <p class="text-2xl font-bold text-blue-600" id="totalTransfers">$0.00</p>
                        </div>
                        <div class="p-3 bg-blue-100 rounded-full">
                            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 md:p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Total Trades</p>
                            <p class="text-2xl font-bold text-purple-600" id="totalTrades">0</p>
                        </div>
                        <div class="p-3 bg-purple-100 rounded-full">
                            <svg class="w-6 h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Transactions Table (Desktop) -->
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden hidden md:block">
                <div class="p-6 border-b border-gray-200">
                    <h2 class="text-xl font-bold text-gray-800">Transaction History</h2>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Type</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Asset</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Amount</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">USD Value</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Date</th>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Details</th>
                            </tr>
                        </thead>
                        <tbody id="transactionsTableBody" class="bg-white divide-y divide-gray-200">
                            <?php if (empty($transactions)): ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-4 text-center text-gray-500">Henüz işlem geçmişi bulunmamaktadır.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($transactions as $tx): 
                                    $type_map = [
                                        'deposit' => 'Yatır',
                                        'withdraw' => 'Çek',
                                        'transfer' => 'Aktar',
                                        'trade' => 'İşlem',
                                        'staking' => 'Stake',
                                        'reward' => 'Ödül'
                                    ];
                                    $status_map = [
                                        'pending' => 'Beklemede',
                                        'completed' => 'Tamamlandı',
                                        'failed' => 'Başarısız',
                                        'cancelled' => 'İptal Edildi'
                                    ];
                                    $status_color = [
                                        'pending' => 'yellow',
                                        'completed' => 'green',
                                        'failed' => 'red',
                                        'cancelled' => 'gray'
                                    ];
                                ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium <?php echo $tx['type'] === 'deposit' || $tx['type'] === 'trade' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                            <?php echo $type_map[$tx['type']] ?? ucfirst($tx['type']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900 font-mono"><?php echo htmlspecialchars($tx['coin_symbol']); ?></td>
                                    <td class="px-6 py-4 text-right text-sm text-gray-900 font-mono"><?php echo format_tx_amount($tx['coin_symbol'], $tx['amount']); ?></td>
                                    <td class="px-6 py-4 text-right text-sm text-gray-900">$<?php echo number_format($tx['total'], 2); ?></td>
                                    <td class="px-6 py-4">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-<?php echo $status_color[$tx['status']] ?? 'gray'; ?>-100 text-<?php echo $status_color[$tx['status']] ?? 'gray'; ?>-800">
                                            <?php echo $status_map[$tx['status']] ?? ucfirst($tx['status']); ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-500"><?php echo date('d.m.Y H:i', strtotime($tx['created_at'])); ?></td>
                                    <td class="px-6 py-4">
                                        <?php if ($tx['tx_hash']): ?>
                                            <a href="#" class="text-blue-600 hover:text-blue-800 text-xs" onclick="copyTxHash('<?php echo htmlspecialchars($tx['tx_hash']); ?>')">
                                                <?php echo htmlspecialchars(substr($tx['tx_hash'], 0, 10)) . '...'; ?>
                                            </a>
                                        <?php else: ?>
                                            <span class="text-gray-400 text-xs">-</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Transactions (Mobile Cards) -->
            <div class="md:hidden">
                <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                    <div class="p-4 border-b border-gray-200">
                        <h2 class="text-lg font-bold text-gray-800">Geçmiş</h2>
                    </div>
                    <div id="mobileTxList" class="divide-y divide-gray-100">
                        <?php if (empty($transactions)): ?>
                            <div class="p-6 text-center text-gray-500">Henüz işlem geçmişi bulunmamaktadır.</div>
                        <?php else: ?>
                            <?php foreach ($transactions as $tx): 
                                $type_map = [
                                    'deposit' => 'Yatır',
                                    'withdraw' => 'Çek',
                                    'transfer' => 'Aktar',
                                    'trade' => 'İşlem',
                                    'staking' => 'Stake',
                                    'reward' => 'Ödül'
                                ];
                                $status_map = [
                                    'pending' => 'Beklemede',
                                    'completed' => 'Tamamlandı',
                                    'failed' => 'Başarısız',
                                    'cancelled' => 'İptal Edildi'
                                ];
                                $status_color = [
                                    'pending' => 'yellow',
                                    'completed' => 'green',
                                    'failed' => 'red',
                                    'cancelled' => 'gray'
                                ];
                                $statusBadge = 'bg-' . ($status_color[$tx['status']] ?? 'gray') . '-100 text-' . ($status_color[$tx['status']] ?? 'gray') . '-800';
                            ?>
                            <div class="p-4">
                                <div class="flex items-start justify-between gap-3">
                                    <div>
                                        <div class="text-sm font-semibold text-gray-800"><?php echo $type_map[$tx['type']] ?? ucfirst($tx['type']); ?> • <?php echo htmlspecialchars($tx['coin_symbol']); ?></div>
                                        <div class="mt-1 text-xs text-gray-500"><?php echo date('d.m.Y H:i', strtotime($tx['created_at'])); ?></div>
                                    </div>
                                    <span class="px-2 py-0.5 text-xs font-semibold rounded-full <?php echo $statusBadge; ?>"><?php echo $status_map[$tx['status']] ?? ucfirst($tx['status']); ?></span>
                                </div>
                                <div class="mt-3 grid grid-cols-3 gap-2 text-xs text-gray-600">
                                    <div><span class="text-gray-500">Miktar:</span> <?php echo format_tx_amount($tx['coin_symbol'], $tx['amount']); ?></div>
                                    <div class="text-right col-span-2"><span class="text-gray-500">USD:</span> $<?php echo number_format($tx['total'], 2); ?></div>
                                </div>
                                <div class="mt-2 text-xs text-gray-500 break-all">
                                    <?php if ($tx['tx_hash']): ?>
                                        Tx: <button class="underline" onclick="copyTxHash('<?php echo htmlspecialchars($tx['tx_hash']); ?>')"><?php echo htmlspecialchars(substr($tx['tx_hash'], 0, 18)) . '...'; ?></button>
                                    <?php else: ?>
                                        <span class="text-gray-400">-</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        let transactionsData = <?php echo json_encode($transactions); ?>;
        let filteredData = [];

        // Copy transaction hash to clipboard
        function copyTxHash(hash) {
            navigator.clipboard.writeText(hash);
            alert('İşlem hash\'i kopyalandı!');
        }

        // Initialize with real data
        filteredData = transactionsData;

        // Normalize PHP transactions to frontend format
        function normalizeTransactions(phpTxs) {
            return (phpTxs || []).map(tx => ({
                type: tx.type || 'unknown',
                asset: tx.coin_symbol || 'UNKNOWN',
                amount: parseFloat(tx.amount || 0),
                usdValue: parseFloat(tx.total || 0),
                status: tx.status || 'completed',
                date: new Date(tx.created_at || Date.now()),
                details: tx.details || '',
                txHash: tx.tx_hash || ''
            }));
        }

        function updateTransactionsTable() {
            const tbody = document.getElementById('transactionsTableBody');
            tbody.innerHTML = '';

            filteredData.forEach(transaction => {
                const typeColor = getTypeColor(transaction.type);
                const statusColor = getStatusColor(transaction.status);
                const typeIcon = getTypeIcon(transaction.type);

                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-2">
                            <div class="p-1 rounded-full ${typeColor.bg}">
                                ${typeIcon}
                            </div>
                            <span class="text-sm font-medium text-gray-800 capitalize">${transaction.type}</span>
                        </div>
                    </td>
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-2">
                            <div class="w-6 h-6 rounded-full bg-gray-100 flex items-center justify-center">
                                <span class="text-xs font-semibold">${transaction.asset.charAt(0)}</span>
                            </div>
                            <span class="text-sm font-medium text-gray-800">${transaction.asset}</span>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${transaction.amount.toFixed(8)}</td>
                    <td class="px-6 py-4 text-right text-gray-800 font-semibold">$${transaction.usdValue.toFixed(2)}</td>
                    <td class="px-6 py-4">
                        <span class="px-2 py-1 text-xs font-semibold rounded-full ${statusColor.bg} ${statusColor.text}">
                            ${transaction.status}
                        </span>
                    </td>
                    <td class="px-6 py-4 text-sm text-gray-600">${transaction.date.toLocaleDateString()} ${transaction.date.toLocaleTimeString()}</td>
                    <td class="px-6 py-4">
                        <div class="text-sm text-gray-600">${transaction.details}</div>
                        <div class="text-xs text-gray-400 font-mono">${transaction.txHash}</div>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateSummary() {
            let totalDeposits = 0;
            let totalWithdrawals = 0;
            let totalTransfers = 0;
            let totalTrades = 0;

            filteredData.forEach(transaction => {
                switch (transaction.type) {
                    case 'deposit':
                        totalDeposits += transaction.usdValue;
                        break;
                    case 'withdraw':
                        totalWithdrawals += transaction.usdValue;
                        break;
                    case 'transfer':
                        totalTransfers += transaction.usdValue;
                        break;
                    case 'trade':
                        totalTrades++;
                        break;
                }
            });

            document.getElementById('totalDeposits').textContent = `$${totalDeposits.toFixed(2)}`;
            document.getElementById('totalWithdrawals').textContent = `$${totalWithdrawals.toFixed(2)}`;
            document.getElementById('totalTransfers').textContent = `$${totalTransfers.toFixed(2)}`;
            document.getElementById('totalTrades').textContent = totalTrades.toString();
        }

        function getTypeColor(type) {
            const colors = {
                deposit: { bg: 'bg-green-100', text: 'text-green-600' },
                withdraw: { bg: 'bg-red-100', text: 'text-red-600' },
                transfer: { bg: 'bg-blue-100', text: 'text-blue-600' },
                trade: { bg: 'bg-purple-100', text: 'text-purple-600' }
            };
            return colors[type] || { bg: 'bg-gray-100', text: 'text-gray-600' };
        }

        function getStatusColor(status) {
            const colors = {
                completed: { bg: 'bg-green-100', text: 'text-green-800' },
                pending: { bg: 'bg-yellow-100', text: 'text-yellow-800' },
                failed: { bg: 'bg-red-100', text: 'text-red-800' }
            };
            return colors[status] || { bg: 'bg-gray-100', text: 'text-gray-800' };
        }

        function getTypeIcon(type) {
            const icons = {
                deposit: '<svg class="w-4 h-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg>',
                withdraw: '<svg class="w-4 h-4 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4m16 0l-4-4m4 4l-4 4"></path></svg>',
                transfer: '<svg class="w-4 h-4 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path></svg>',
                trade: '<svg class="w-4 h-4 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path></svg>'
            };
            return icons[type] || '<svg class="w-4 h-4 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path></svg>';
        }

        function applyFilters() {
            const typeFilter = document.getElementById('typeFilter').value;
            const assetFilter = document.getElementById('assetFilter').value;
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;

            filteredData = transactionsData.filter(transaction => {
                const typeMatch = typeFilter === 'all' || transaction.type === typeFilter;
                const assetMatch = assetFilter === 'all' || transaction.asset === assetFilter;
                
                let dateMatch = true;
                if (startDate) {
                    dateMatch = dateMatch && transaction.date >= new Date(startDate);
                }
                if (endDate) {
                    dateMatch = dateMatch && transaction.date <= new Date(endDate + 'T23:59:59');
                }

                return typeMatch && assetMatch && dateMatch;
            });

            updateTransactionsTable();
            updateSummary();
        }

        function resetFilters() {
            document.getElementById('typeFilter').value = 'all';
            document.getElementById('assetFilter').value = 'all';
            document.getElementById('startDate').value = '';
            document.getElementById('endDate').value = '';
            
            filteredData = [...transactionsData];
            updateTransactionsTable();
            updateSummary();
        }

        function exportHistory() {
            if (filteredData.length === 0) {
                alert('No data to export');
                return;
            }

            const csvContent = [
                ['Type', 'Asset', 'Amount', 'USD Value', 'Status', 'Date', 'Details', 'Transaction Hash'],
                ...filteredData.map(transaction => [
                    transaction.type,
                    transaction.asset,
                    transaction.amount,
                    transaction.usdValue,
                    transaction.status,
                    transaction.date.toISOString(),
                    transaction.details,
                    transaction.txHash
                ])
            ].map(row => row.join(',')).join('\n');

            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'wallet_history.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }

        // Initialize on page load with REAL data
        document.addEventListener('DOMContentLoaded', function() {
            if (Array.isArray(transactionsData) && transactionsData.length > 0) {
                filteredData = normalizeTransactions(transactionsData);
                updateTransactionsTable();
                updateSummary();
            } else {
                // No transactions; keep server-rendered empty state
            }
        });
    </script>

    <?php include 'components/footer.php'; ?>
</body>
</html>


