<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

require_once 'config/database.php';

$user_id = $_SESSION['user_id'];

// Kullanıcının cüzdan verilerini veritabanından çek
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallets = $stmt->fetchAll();
} catch (Exception $e) {
    $wallets = [];
}

// Kripto para fiyatlarını yükle
$crypto_prices = [];
if (file_exists('assets/crypto_prices.json')) {
    $price_data = json_decode(file_get_contents('assets/crypto_prices.json'), true);
    if (isset($price_data['prices'])) {
        $crypto_prices = $price_data['prices'];
    }
}

// Toplam bakiyeleri hesapla
$total_balance = 0;
$available_balance = 0;
$in_orders = 0;

foreach ($wallets as $wallet) {
    $coin_symbol = $wallet['coin_symbol'];
    $price = 1; // Default price
    
    if (isset($crypto_prices[$coin_symbol])) {
        $price = floatval($crypto_prices[$coin_symbol]['raw_price']);
    }
    
    $total_balance += $wallet['balance'] * $price;
    $available_balance += $wallet['available'] * $price;
    $in_orders += $wallet['frozen'] * $price;
}

include 'components/header.php';
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cüzdan - KriptoCum</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/style.css">
    <script src="assets/js/dropdown.js"></script>
</head>
<body class="bg-gray-50">
    <!-- Main Content -->
    <div class="pt-16 min-h-screen">
        <div class="container mx-auto px-4 py-8">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">Cüzdan</h1>
                <p class="text-gray-600">Kripto para varlıklarınızı yönetin</p>
                <button onclick="updatePrices()" class="mt-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 text-sm">
                    🔄 Manuel Güncelle
                </button>
            </div>

            <!-- Wallet Overview -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                <!-- Total Balance -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Toplam Bakiye</p>
                            <p class="text-2xl font-bold text-gray-900" id="totalBalance">$<?php echo number_format($total_balance, 2); ?></p>
                        </div>
                        <div class="p-3 bg-blue-100 rounded-full">
                            <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- Available Balance -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Müsait</p>
                            <p class="text-2xl font-bold text-gray-900" id="availableBalance">$<?php echo number_format($available_balance, 2); ?></p>
                        </div>
                        <div class="p-3 bg-green-100 rounded-full">
                            <svg class="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>

                <!-- In Orders -->
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Emirlerde</p>
                            <p class="text-2xl font-bold text-gray-900" id="inOrders">$<?php echo number_format($in_orders, 2); ?></p>
                        </div>
                        <div class="p-3 bg-yellow-100 rounded-full">
                            <svg class="w-6 h-6 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="flex flex-wrap gap-4 mb-8">
                <button onclick="showDepositModal()" class="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                    </svg>
                    Yatır
                </button>
                <button onclick="showWithdrawModal()" class="px-6 py-3 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 12H4m16 0l-4-4m4 4l-4 4"></path>
                    </svg>
                    Çek
                </button>
                <button onclick="showTransferModal()" class="px-6 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path>
                    </svg>
                    Aktar
                </button>
                <button onclick="showHistory()" class="px-6 py-3 bg-gray-600 text-white rounded-lg hover:bg-gray-700 transition-colors flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Geçmiş
                </button>
            </div>

            <!-- Assets Table -->
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                <div class="p-6 border-b border-gray-200">
                    <div class="flex items-center justify-between">
                        <h2 class="text-xl font-bold text-gray-800">Varlıklar</h2>
                        <div class="flex items-center gap-4">
                            <input type="text" placeholder="🔍 Varlık ara..." class="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" id="searchInput">
                            <button onclick="showHideZero()" class="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50">
                                <span id="hideZeroText">Sıfırları Gizle</span>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-semibold text-gray-600 uppercase">Varlık</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Toplam</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Müsait</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">Emirlerde</th>
                                <th class="px-6 py-3 text-right text-xs font-semibold text-gray-600 uppercase">USD Değeri</th>
                                <th class="px-6 py-3 text-center text-xs font-semibold text-gray-600 uppercase">İşlemler</th>
                            </tr>
                        </thead>
                        <tbody id="assetsTableBody" class="bg-white divide-y divide-gray-200">
                            <!-- Assets will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Deposit Modal -->
    <div id="depositModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Yatır</h3>
                    <button onclick="hideDepositModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Varlık Seç</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="BTC">Bitcoin (BTC)</option>
                            <option value="ETH">Ethereum (ETH)</option>
                            <option value="USDT">Tether (USDT)</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Deposit Address</label>
                        <div class="flex">
                            <input type="text" readonly value="1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa" class="flex-1 px-3 py-2 border border-gray-300 rounded-l-lg bg-gray-50">
                            <button class="px-3 py-2 bg-blue-600 text-white rounded-r-lg hover:bg-blue-700">Copy</button>
                        </div>
                    </div>
                    <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-3">
                        <p class="text-sm text-yellow-800">⚠️ Only send BTC to this address. Sending other assets may result in permanent loss.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Withdraw Modal -->
    <div id="withdrawModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Withdraw</h3>
                    <button onclick="hideWithdrawModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Varlık Seç</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="BTC">Bitcoin (BTC)</option>
                            <option value="ETH">Ethereum (ETH)</option>
                            <option value="USDT">Tether (USDT)</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Withdraw Address</label>
                        <input type="text" placeholder="Enter withdrawal address" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Amount</label>
                        <input type="number" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <button class="w-full bg-red-600 text-white py-2 rounded-lg hover:bg-red-700">Withdraw</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Transfer Modal -->
    <div id="transferModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
        <div class="flex items-center justify-center min-h-screen p-4">
            <div class="bg-white rounded-lg max-w-md w-full p-6">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-800">Transfer</h3>
                    <button onclick="hideTransferModal()" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">From</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="spot">Spot Wallet</option>
                            <option value="futures">Futures Wallet</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">To</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="futures">Futures Wallet</option>
                            <option value="spot">Spot Wallet</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Asset</label>
                        <select class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="USDT">USDT</option>
                            <option value="BTC">BTC</option>
                            <option value="ETH">ETH</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Amount</label>
                        <input type="number" placeholder="0.00" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    <button class="w-full bg-green-600 text-white py-2 rounded-lg hover:bg-green-700">Transfer</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Veritabanından gelen cüzdan verileri
        let walletsData = <?php echo json_encode($wallets); ?>;
        const cryptoPrices = <?php echo json_encode($crypto_prices); ?>;
        
        let hideZero = false;
        let assetsData = [];

        function loadAssets() {
            // Veritabanından gelen verileri işle
            assetsData = walletsData.map(wallet => {
                const coin_symbol = wallet.coin_symbol;
                const price = cryptoPrices[coin_symbol] ? parseFloat(cryptoPrices[coin_symbol].raw_price) : 1;
                
                return {
                    symbol: coin_symbol,
                    name: getCoinName(coin_symbol),
                    total: parseFloat(wallet.balance),
                    available: parseFloat(wallet.available),
                    inOrders: parseFloat(wallet.frozen),
                    price: price
                };
            });
            
            updateAssetsTable();
            updateWalletOverview();
        }

        // Gerçek zamanlı fiyat ve cüzdan güncellemesi
        function updatePrices() {
            console.log('updatePrices() fonksiyonu çağrıldı');
            
            // Sayfa yenileme ile güncelle (daha güvenli)
            location.reload();
        }
        
        // Global fonksiyon olarak tanımla
        window.updatePrices = updatePrices;

        function getCoinName(symbol) {
            const names = {
                'BTC': 'Bitcoin',
                'ETH': 'Ethereum',
                'USDT': 'Tether',
                'USDC': 'USD Coin',
                'SOL': 'Solana',
                'XRP': 'Ripple',
                'ADA': 'Cardano',
                'DOT': 'Polkadot',
                'MATIC': 'Polygon',
                'AVAX': 'Avalanche'
            };
            return names[symbol] || symbol;
        }

        function updateAssetsTable() {
            const tbody = document.getElementById('assetsTableBody');
            tbody.innerHTML = '';

            const filteredAssets = hideZero ? assetsData.filter(asset => asset.total > 0) : assetsData;

            filteredAssets.forEach(asset => {
                const usdValue = asset.total * asset.price;
                const tr = document.createElement('tr');
                tr.className = 'hover:bg-gray-50';
                tr.innerHTML = `
                    <td class="px-6 py-4">
                        <div class="flex items-center gap-3">
                            <div class="w-8 h-8 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden">
                                <img src="assets/kriptologolari/${asset.symbol.toLowerCase()}.png" 
                                     alt="${asset.symbol}" 
                                     class="w-full h-full object-cover"
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="w-full h-full bg-gray-100 flex items-center justify-center" style="display: none;">
                                    <span class="text-xs font-semibold">${asset.symbol.charAt(0)}</span>
                                </div>
                            </div>
                            <div>
                                <div class="font-semibold text-gray-800">${asset.symbol}</div>
                                <div class="text-xs text-gray-500">${asset.name}</div>
                            </div>
                        </div>
                    </td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${asset.total.toFixed(8)}</td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${asset.available.toFixed(8)}</td>
                    <td class="px-6 py-4 text-right text-gray-800 font-mono">${asset.inOrders.toFixed(8)}</td>
                    <td class="px-6 py-4 text-right text-gray-800 font-semibold">$${usdValue.toFixed(2)}</td>
                    <td class="px-6 py-4 text-center">
                        <div class="flex justify-center gap-2">
                            <a href="trade.php" target="_blank" class="text-blue-600 hover:text-blue-700 text-sm">Trade</a>
                            <button onclick="depositAsset('${asset.symbol}')" class="text-green-600 hover:text-green-700 text-sm">Deposit</button>
                            <button onclick="withdrawAsset('${asset.symbol}')" class="text-red-600 hover:text-red-700 text-sm">Withdraw</button>
                        </div>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function updateWalletOverview() {
            let totalBalance = 0;
            let availableBalance = 0;
            let inOrders = 0;

            assetsData.forEach(asset => {
                const usdValue = asset.total * asset.price;
                const availableValue = asset.available * asset.price;
                const ordersValue = asset.inOrders * asset.price;

                totalBalance += usdValue;
                availableBalance += availableValue;
                inOrders += ordersValue;
            });

            document.getElementById('totalBalance').textContent = `$${totalBalance.toFixed(2)}`;
            document.getElementById('availableBalance').textContent = `$${availableBalance.toFixed(2)}`;
            document.getElementById('inOrders').textContent = `$${inOrders.toFixed(2)}`;
        }
        
        
        // Gerçek zamanlı güncelleme başlat
        function startRealTimeUpdates() {
            // Her 60 saniyede bir sayfayı yenile
            setInterval(() => {
                console.log('Otomatik güncelleme...');
                location.reload();
            }, 60000);
        }
        
        // Trade sayfasından gelen bildirimleri dinle
        window.addEventListener('message', function(event) {
            if (event.data.type === 'WALLET_UPDATE') {
                console.log('Wallet güncelleme bildirimi alındı');
                // Sayfayı yenile
                setTimeout(() => {
                    location.reload();
                }, 1000);
            }
        });
        
        // Sayfa yüklendiğinde başlat
        document.addEventListener('DOMContentLoaded', function() {
            loadAssets();
            startRealTimeUpdates();
        });
            hideZero = !hideZero;
            document.getElementById('hideZeroText').textContent = hideZero ? 'Show Zero' : 'Hide Zero';
            updateAssetsTable();
        }

        function showDepositModal() {
            document.getElementById('depositModal').classList.remove('hidden');
        }

        function hideDepositModal() {
            document.getElementById('depositModal').classList.add('hidden');
        }

        function showWithdrawModal() {
            document.getElementById('withdrawModal').classList.remove('hidden');
        }

        function hideWithdrawModal() {
            document.getElementById('withdrawModal').classList.add('hidden');
        }

        function showTransferModal() {
            document.getElementById('transferModal').classList.remove('hidden');
        }

        function hideTransferModal() {
            document.getElementById('transferModal').classList.add('hidden');
        }

        function showHistory() {
            window.location.href = 'wallet-history.php';
        }

        function depositAsset(symbol) {
            // Trade sayfasına yönlendir
            window.open(`trade.php?symbol=${symbol}`, '_blank');
        }

        function withdrawAsset(symbol) {
            // Trade sayfasına yönlendir
            window.open(`trade.php?symbol=${symbol}`, '_blank');
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadAssets();
            startRealTimeUpdates();
        });
    </script>

    <?php include 'components/footer.php'; ?>
</body>
</html>


